<?php
if ( ! defined( 'ABSPATH' ) ) exit;

// Handle cancel action
if ( isset($_GET['action'], $_GET['order_id'], $_GET['item_id'], $_GET['unit']) && $_GET['action'] === 'cancel_subscription' ) {
    $easysubs_order_id = absint($_GET['order_id']);
    $easysubs_item_id  = absint($_GET['item_id']);
    $easysubs_unit     = absint($_GET['unit']);
    
    $easysubs_paypal_plan_id = get_post_meta( $easysubs_order_id, '_paypal_subscription_id', true );


    if ( $easysubs_order_id && $easysubs_item_id && $easysubs_unit && current_user_can('manage_woocommerce') && check_admin_referer('cancel_subscription_' . $easysubs_order_id . '_' . $easysubs_item_id . '_' . $easysubs_unit) && empty( $easysubs_paypal_plan_id ) ) {

        // Unschedule only this unit’s actions
        if ( function_exists('as_unschedule_all_actions') ) {
            as_unschedule_all_actions(
                'easy_subscriptions_process_payment',
                [
                    'order_id' => (int) $easysubs_order_id,
                    'item_id'  => (int) $easysubs_item_id,
                    'unit'     => (int) $easysubs_unit,
                ]
            );
        }

        // Mark this specific unit as cancelled (store at item+unit level)
        wc_update_order_item_meta( $easysubs_item_id, '_easy_subscriptions_status_unit_' . $easysubs_unit, 'cancelled' );
        wc_delete_order_item_meta( $easysubs_item_id, '_easy_subscriptions_next_payment_unit_' . $easysubs_unit );

        // Redirect back to table view
        wp_safe_redirect( remove_query_arg( ['action','order_id','item_id','unit','_wpnonce'] ) );
        exit;
} else if ( ! empty( $easysubs_paypal_plan_id ) ) {
    // Make sure gateway class is loaded
    if ( class_exists( 'Easy_Subscriptions_Gateway_PayPal' ) ) {
        $easysubs_gateway = new Easy_Subscriptions_Gateway_PayPal();

        // Retrieve the subscription ID you saved earlier
        $easysubs_subscription_id = get_post_meta( $easysubs_order_id, '_paypal_subscription_id', true );
        wc_update_order_item_meta( $easysubs_item_id, '_easy_subscriptions_status_unit_' . $easysubs_unit, 'cancelled' );
        if ( ! empty( $easysubs_subscription_id ) ) {
            try {
                $easysubs_gateway->cancel_subscription( $easysubs_subscription_id );
                // Optionally: update order/subscription status
                update_post_meta( $easysubs_order_id, '_paypal_subscription_status', 'cancelled' );
            } catch ( Exception $e ) {
                //error_log( 'PayPal cancellation failed: ' . $e->getMessage() );
            }
        }
    }
}
}

/**
 * @var WC_Order[] $orders
 * @var int        $paged
 * @var int        $per_page
 * @var int        $total
 * @var int        $max_num_pages
 */


function easy_subs_is_active( $easysubs_order_id ) {
    $easysubs_paypal_plan_id = get_post_meta( $easysubs_order_id, '_paypal_subscription_id', true );
    $paypal_sub_status = get_post_meta( $easysubs_order_id, '_paypal_subscription_status', true );
    if ( ! empty( $easysubs_paypal_plan_id ) && (strtolower( $paypal_sub_status ) !== 'cancelled' )) {
        return 'active';
    }
    // 1. Action status takes precedence
    if ( function_exists( 'as_next_scheduled_action' ) ) {
        $has_future = false;
        $order = wc_get_order( $easysubs_order_id );
        if ( $order ) {
            foreach ( $order->get_items() as $easysubs_item ) {
                $easysubs_item_id = $easysubs_item->get_id();
                $easysubs_qty     = (int) $easysubs_item->get_quantity();
                for ( $easysubs_i = 1; $easysubs_i <= $easysubs_qty; $easysubs_i++ ) {
                    $next_ts = as_next_scheduled_action(
                        'easy_subscriptions_process_payment',
                        [
                            'order_id' => (int) $easysubs_order_id,
                            'item_id'  => (int) $easysubs_item_id,
                            'unit'     => $easysubs_i,
                        ]
                    );
                    if ( $next_ts && $next_ts > time() ) {
                        $has_future = true;
                        break 2; // stop as soon as we find one scheduled job
                    }
                }
            }
        }

        if ( ! $has_future ) {
            // No scheduled actions left → cancelled
            return 'cancelled';
        }
    }

    // 2. Otherwise fall back to stored meta status
    $status = get_post_meta( $easysubs_order_id, '_easy_subscriptions_status', true );

    if ( empty( $status ) ) {
        $items = wc_get_order( $easysubs_order_id )->get_items();
        foreach ( $items as $easysubs_item ) {
            $item_status = wc_get_order_item_meta( $easysubs_item->get_id(), '_easy_subscriptions_status', true );
            if ( $item_status ) {
                $status = $item_status;
                break;
            }
        }
    }

    if ( $status === 'suspended' ) return 'suspended';
    if ( $status === 'cancelled' ) return 'cancelled';
    if ( $status === 'active' ) return 'active';
    if ( $status === 'failed' ) return 'failed';

    // 3. Fallback to next_payment meta
    $next_meta = (int) get_post_meta( $easysubs_order_id, '_easy_subscriptions_next_payment', true );
    if ( $next_meta && $next_meta > time() ) {
        return 'active';
    }

    return 'cancelled';
}

/**
 * Determine subscription status for a specific unit of an order item.
 *
 * @param int $easysubs_order_id
 * @param int $easysubs_item_id
 * @param int $easysubs_unit
 * @return string One of 'active', 'cancelled', 'suspended', 'failed'
 */
function easy_subs_unit_status( $easysubs_order_id, $easysubs_item_id, $easysubs_unit ) {
    // 1. Action Scheduler takes precedence (is there a future job?)

    $easysubs_paypal_plan_id = get_post_meta( $easysubs_order_id, '_paypal_subscription_id', true );
    $paypal_sub_status = get_post_meta( $easysubs_order_id, '_paypal_subscription_status', true );

    $easysubs_unit_status = wc_get_order_item_meta( $easysubs_item_id, '_easy_subscriptions_status_unit_' . $easysubs_unit, true );

    if ( ! empty( $easysubs_paypal_plan_id ) && (strtolower( $paypal_sub_status ) !== 'cancelled' )) {
        return 'active';
    }
    if ( function_exists( 'as_next_scheduled_action' ) ) {
        $next_ts = as_next_scheduled_action(
            'easy_subscriptions_process_payment',
            [
                'order_id' => (int) $easysubs_order_id,
                'item_id'  => (int) $easysubs_item_id,
                'unit'     => (int) $easysubs_unit,
            ]
        );

        if ( ! $next_ts || $next_ts <= time() ) {
            if ($easysubs_unit_status == 'suspended'){
                 return 'suspended';
            }
            // No scheduled future action → treat as cancelled
            return 'cancelled';
        }

    }

    // 2. Check per-unit meta
    if ( $easysubs_unit_status ) {
        return $easysubs_unit_status;
    }
    // 3. Fallback: check item-level meta
    $item_status = wc_get_order_item_meta( $easysubs_item_id, '_easy_subscriptions_status', true );
    if ( $item_status ) {
        return $item_status;
    }

    // 4. Fallback: check order-level meta
    $order_status = get_post_meta( $easysubs_order_id, '_easy_subscriptions_status', true );
    if ( $order_status ) {
        return $order_status;
    }

    // 5. Fallback: check next payment date
    $next_meta = (int) wc_get_order_item_meta( $easysubs_item_id, '_easy_subscriptions_next_payment_u' . $easysubs_unit, true );
    if ( ! $next_meta ) {
        $next_meta = (int) wc_get_order_item_meta( $easysubs_item_id, '_easy_subscriptions_next_payment', true );
    }
    if ( $next_meta && $next_meta > time() ) {
        return 'active';
    }

    return 'cancelled';
}

function easy_subs_completed_renewals( $easysubs_order_id ) {
    // Count completed actions for this order (renewal runs)
    if ( function_exists( 'as_get_scheduled_actions' ) ) {
        $completed = as_get_scheduled_actions( [
            'hook'     => 'easy_subscriptions_process_payment',
            'status'   => 'complete',
            'args'     => [ 'order_id' => (int) $easysubs_order_id ],
            'per_page' => 1000,     // large enough for practical counts
        ] );
        if ( is_array( $completed ) ) {
            return count( $completed );
        }
    }
    // Optional fallback to a meta counter if you decide to persist it
    return (int) get_post_meta( $easysubs_order_id, '_easy_subscriptions_renewals', true );
}
?>
<div class="wrap woocommerce">
    <h1 class="wp-heading-inline"><?php esc_html_e( 'Easy Subscriptions', 'easy-subscriptions' ); ?></h1>
    <hr class="wp-header-end">

    <?php
    // Count totals
    $easysubs_total_all = count($orders);
    $easysubs_total_active = 0;
    $easysubs_total_cancelled = 0;
    $easysubs_total_failed = 0;
    foreach ( $orders as $order ) {
        if ( ! $order instanceof WC_Order ) { continue; }
        $easysubs_active = easy_subs_is_active( $order->get_id() );
        if ( $easysubs_active === 'active' ) {
            $easysubs_total_active++;
        } elseif ( $easysubs_active === 'cancelled' ) {
            $easysubs_total_cancelled++;
        } elseif ( $easysubs_active === 'failed' ) {
            $easysubs_total_failed++;
        }
    }

    // Determine current filter
    $easysubs_current_filter = isset($_GET['subscription_status']) ? sanitize_text_field( wp_unslash($_GET['subscription_status'])) : 'all';

    // Build base URL
    $easysubs_base_url = remove_query_arg( ['subscription_status', 'paged'] );
    ?>

    <ul class="subsubsub">
        <li class="all">
            <a href="<?php echo esc_url( $easysubs_base_url ); ?>" class="<?php echo $easysubs_current_filter === 'all' ? 'current' : ''; ?>">
                <?php  
                // translators: %d is the total number of all subscriptions.
                printf( wp_kses_post( __( 'All <span class="count">(%s)</span>', 'easy-subscriptions' ) ), esc_html( $easysubs_total_all ) );?>
            </a> |
        </li>
        <li class="active">
            <a href="<?php echo esc_url( add_query_arg( 'subscription_status', 'active', $easysubs_base_url ) ); ?>" class="<?php echo $easysubs_current_filter === 'active' ? 'current' : ''; ?>">
                <?php 
                // translators: %s is the total number of active subscriptions.
                printf( esc_html__( 'Active (%s)', 'easy-subscriptions' ), esc_html( $easysubs_total_active ) );
                ?>
            </a> |
        </li>
        <li class="cancelled">
            <a href="<?php echo esc_url( add_query_arg( 'subscription_status', 'cancelled', $easysubs_base_url ) ); ?>" class="<?php echo $easysubs_current_filter === 'cancelled' ? 'current' : ''; ?>">
                <?php 
                // translators: %s is the total number of cancelled subscriptions.
                printf( esc_html__( 'Cancelled (%s)', 'easy-subscriptions' ), esc_html( $easysubs_total_cancelled ) );
                ?>
            </a> |
        </li>
        <li class="failed">
            <a href="<?php echo esc_url( add_query_arg( 'subscription_status', 'failed', $easysubs_base_url ) ); ?>" class="<?php echo $easysubs_current_filter === 'failed' ? 'current' : ''; ?>">
                <?php 
                // translators: %s is the total number of failed subscriptions.
                printf( esc_html__( 'Failed (%s)', 'easy-subscriptions' ), esc_html( $easysubs_total_failed ) );
                ?>
            </a>
        </li>
    </ul>
    <br class="clear" />

    <form method="get">
        <input type="hidden" name="page" value="easy-subscriptions" />
        <?php wp_nonce_field( 'easy_subscriptions_search', '_easy_subs_nonce' ); ?>
        <input type="search" name="s" value="<?php echo isset( $_GET['s'] ) ? esc_attr( sanitize_text_field( wp_unslash( $_GET['s'] ) ) ) : ''; ?>" placeholder="<?php esc_attr_e( 'Search subscriptions…', 'easy-subscriptions' ); ?>" />
        <button type="submit" class="button"><?php esc_html_e( 'Search', 'easy-subscriptions' ); ?></button>
    </form>
    <div style="margin-bottom:15px;"></div>

    <table class="wp-list-table widefat fixed striped shop_order">
        <thead>
            <tr>
                <th class="column-primary"><?php esc_html_e( 'Status', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Subscription', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Items', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Recurring Total', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Renewals', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Failed Attempts', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Date', 'easy-subscriptions' ); ?></th>
                <th><?php esc_html_e( 'Actions', 'easy-subscriptions' ); ?></th>
            </tr>
        </thead>
        <tbody>
        <?php if ( ! empty( $orders ) ) : ?>
            <?php foreach ( $orders as $order ) :
                if ( ! $order instanceof WC_Order ) { continue; }
                $easysubs_order_id = $order->get_id();

                // Active/Cancelled status as requested (green/red “pill” like Woo orders)
                $easysubs_active = easy_subs_is_active( $easysubs_order_id );

                if ( $easysubs_current_filter === 'active' && $easysubs_active !== 'active' ) continue;
                if ( $easysubs_current_filter === 'cancelled' && $easysubs_active === 'active' ) continue;
                if ( $easysubs_current_filter === 'failed' && $easysubs_active !== 'failed' ) continue;

                // Subscription number = order number
                $easysubs_sub_number = $order->get_order_number();

                // Each subscription item in this order should render as its own "subscription row"
foreach ( $order->get_items() as $easysubs_item ) {
    $easysubs_pid    = $easysubs_item->get_variation_id() ?: $easysubs_item->get_product_id();
    $easysubs_is_sub = get_post_meta( $easysubs_pid, '_easy_subscriptions', true );
    if ( $easysubs_is_sub !== 'yes' ) {
        continue;
    }

    $easysubs_item_name   = $easysubs_item->get_name();
    $easysubs_qty         = (int) $easysubs_item->get_quantity();
    $easysubs_unit_price  = $easysubs_qty > 0 ? $easysubs_item->get_total() / $easysubs_qty : 0;

    // Subscription cadence from item meta first, fallback to product
    $easysubs_interval = wc_get_order_item_meta( $easysubs_item->get_id(), '_easy_subscriptions_period_interval', true )
        ?: get_post_meta( $easysubs_pid, '_easy_subscriptions_period_interval', true );
    $easysubs_unit     = wc_get_order_item_meta( $easysubs_item->get_id(), '_easy_subscriptions_period_unit', true )
        ?: get_post_meta( $easysubs_pid, '_easy_subscriptions_period_unit', true );

    // Try to load original price saved at checkout
    $easysubs_original_price = wc_get_order_item_meta( $easysubs_item->get_id(), '_easy_subscriptions_original_price', true );

    // Fallback: if not saved, use subtotal per unit
    if ( ! $easysubs_original_price ) {
        $easysubs_original_price = $easysubs_item->get_subtotal() / max( 1, $easysubs_qty );
    }

    // Pick whichever is larger
    $easysubs_display_price = max( (float) $easysubs_unit_price, (float) $easysubs_original_price );

    $easysubs_recurring_total = wc_price( $easysubs_display_price );
    if ( $easysubs_interval && $easysubs_unit ) {
        $easysubs_recurring_total .= sprintf(
            ' / %s %s',
            esc_html( $easysubs_interval ),
            esc_html( $easysubs_unit )
        );
    }

    $easysubs_renewals        = easy_subs_completed_renewals( $easysubs_order_id );
    $easysubs_failed_attempts = get_post_meta( $easysubs_order_id, '_easy_subscriptions_failed_attempts', true );
    $easysubs_failed_attempts = is_numeric( $easysubs_failed_attempts ) ? (int) $easysubs_failed_attempts : 0;
    $easysubs_edit_link       = get_edit_post_link( $easysubs_order_id );

    // Expand to one row per unit
    for ( $easysubs_i = 1; $easysubs_i <= $easysubs_qty; $easysubs_i++ ) :
        $easysubs_unit_status = easy_subs_unit_status( $easysubs_order_id, $easysubs_item->get_id(), $easysubs_i );
        ?>
        <tr>
            <!-- Status -->
            <td class="column-primary">
                <?php if ( $easysubs_unit_status === 'active' ) : ?>
                    <mark class="order-status status-processing tips"><?php esc_html_e( 'Active', 'easy-subscriptions' ); ?></mark>
                <?php elseif ( $easysubs_unit_status === 'suspended' ) : ?>
                    <mark class="order-status status-on-hold tips"><?php esc_html_e( 'Suspended', 'easy-subscriptions' ); ?></mark>
                <?php elseif ( $easysubs_unit_status === 'failed' ) : ?>
                    <mark class="order-status status-failed tips"><?php esc_html_e( 'Failed', 'easy-subscriptions' ); ?></mark>
                <?php else : ?>
                    <mark class="order-status status-cancelled tips"><?php esc_html_e( 'Cancelled', 'easy-subscriptions' ); ?></mark>
                <?php endif; ?>
                <button type="button" class="toggle-row" aria-expanded="false">
                    <span class="screen-reader-text"><?php esc_html_e( 'Show more details', 'easy-subscriptions' ); ?></span>
                </button>
            </td>

            <!-- Subscription -->
            <td>
                <?php if ( $easysubs_edit_link ) : ?>
                    <a href="<?php echo esc_url( $easysubs_edit_link ); ?>">#<?php echo esc_html( $order->get_order_number() ); ?></a>
                <?php else : ?>
                    #<?php echo esc_html( $order->get_order_number() ); ?>
                <?php endif; ?>
            </td>

            <!-- Item (with unit index if >1) -->
            <td>
                <?php echo esc_html( $easysubs_item_name ); ?>
                <?php if ( $easysubs_qty > 1 ) echo ' (#' . esc_html( $easysubs_i ) . ')'; ?>
            </td>

            <!-- Recurring Total -->
            <td><?php echo wp_kses_post( $easysubs_recurring_total ); ?></td>

            <!-- Renewals -->
            <td><?php echo esc_html( $easysubs_renewals ); ?></td>

            <!-- Failed Attempts -->
            <td><?php echo $easysubs_failed_attempts > 0 ? esc_html( $easysubs_failed_attempts ) : '-'; ?></td>

            <!-- Date -->
            <td><?php echo esc_html( $order->get_date_created()->date_i18n( get_option('date_format') ) ); ?></td>

            <!-- Actions -->
            <td>
                <?php if ( $easysubs_active === 'active' ) :
                    $easysubs_cancel_url = wp_nonce_url(
                        add_query_arg(
                            [
                                'action'   => 'cancel_subscription',
                                'order_id' => $easysubs_order_id,
                                'item_id'  => $easysubs_item->get_id(),
                                'unit'     => $easysubs_i, // NEW: track specific unit
                            ]
                        ),
                        'cancel_subscription_' . $easysubs_order_id . '_' . $easysubs_item->get_id() . '_' . $easysubs_i
                    );
                ?>
                    <a href="<?php echo esc_url( $easysubs_cancel_url ); ?>" class="button"
                       onclick="return confirm('<?php echo esc_js( __( 'Are you sure you want to cancel this subscription?', 'easy-subscriptions' ) ); ?>');">
                        <?php esc_html_e( 'Cancel', 'easy-subscriptions' ); ?>
                    </a>
                <?php else: ?>
                    —
                <?php endif; ?>
            </td>
        </tr>
        <?php
    endfor;
}
            endforeach; ?>
        <?php else : ?>
            <tr>
                <td colspan="8"><?php esc_html_e( 'No subscriptions found.', 'easy-subscriptions' ); ?></td>
            </tr>
        <?php endif; ?>
        </tbody>
    </table>

<?php
// Simple pagination like core list tables
if ( ! empty( $max_num_pages ) && $max_num_pages > 1 ) {
    echo '<div class="tablenav"><div class="tablenav-pages">';
    echo wp_kses_post( paginate_links( [
        'base'      => add_query_arg( 'paged', '%#%' ),
        'format'    => '',
        'prev_text' => '&laquo;',
        'next_text' => '&raquo;',
        'total'     => $max_num_pages,
        'current'   => $paged,
    ] ) );
    echo '</div></div>';
}
?>
</div>