<?php
if ( ! defined( 'ABSPATH' ) ) exit;
/**
 * Schedule subscription renewals at the per-item + per-unit level.
 */
function easy_subscriptions_schedule( $order_id, $interval, $unit_str ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) {
        return;
    }

    // Calculate base next payment time
    $time = strtotime( "+{$interval} {$unit_str}" );
    update_post_meta( $order_id, '_easy_subscriptions_next_payment', $time );

    // For every subscription item, schedule one job per unit
    foreach ( $order->get_items() as $item_id => $item ) {
        $pid    = $item->get_variation_id() ?: $item->get_product_id();
        $is_sub = get_post_meta( $pid, '_easy_subscriptions', true );
        if ( $is_sub !== 'yes' ) {
            continue;
        }

        $qty = (int) $item->get_quantity();
        for ( $i = 1; $i <= $qty; $i++ ) {
            // Cancel any existing scheduled job for this order/item/unit
            if ( function_exists( 'as_unschedule_all_actions' ) ) {
                as_unschedule_all_actions(
                    'easy_subscriptions_process_payment',
                    [ 'order_id' => (int) $order_id, 'item_id' => (int) $item_id, 'unit' => $i ]
                );
            }

            // Schedule next renewal for this unit
            as_schedule_single_action(
                $time,
                'easy_subscriptions_process_payment',
                [ 'order_id' => (int) $order_id, 'item_id' => (int) $item_id, 'unit' => $i ],
                'easy_subscriptions'
            );
        }
    }
}

/**
 * Processor: run the actual renewal for a single unit.
 * (This replaces the old order-level processor.)
 */
add_action( 'easy_subscriptions_process_payment', function( $order_id, $item_id, $unit ) {
    $order_id = (int) $order_id;
    $item_id  = (int) $item_id;
    $unit     = (int) $unit;

    $handler_lock_key = '_easy_subscriptions_handler_processing_' . $item_id . '_' . $unit;
    if ( ! add_post_meta( $order_id, $handler_lock_key, time(), true ) ) {
        //error_log( "EasySubs: handler lock exists, skipping for #{$order_id} item {$item_id} unit {$unit}" );
        return;
    }

    try {
        $order = wc_get_order( $order_id );
                if ( ! $order ) return;

                $payment_method = $order->get_payment_method();
                $gateway        = null;
//error_log('Payment method is: ' . $payment_method);
                // 🔑 Gateway switch
                switch ( $payment_method ) {
                    case 'woocommerce_payments':
                        $gateway = new Easy_Subscriptions_Gateway_WooPayments();
                        break;
                    case 'stripe':
                    case 'stripe_cc': // sometimes WooCommerce Stripe extension uses this
                        $gateway = new Easy_Subscriptions_Gateway_Stripe();
                        break;

                    case 'paypal':
                    case 'ppcp-gateway': // WooCommerce PayPal Payments plugin
                    case 'paypal_standard': // Old PayPal Standard
                    case 'ppec_paypal':
                        $gateway = new Easy_Subscriptions_Gateway_PayPal();
                        break;

                    default:
                        //error_log( "EasySubs: unsupported payment method {$payment_method} for order {$order_id}" );
                        return;
                }


        $item = $order->get_item( $item_id );
        if ( ! $item ) return;

        $pid    = $item->get_variation_id() ?: $item->get_product_id();
        $is_sub = get_post_meta( $pid, '_easy_subscriptions', true );
        if ( $is_sub !== 'yes' ) return;

        $qty        = max( 1, (int) $item->get_quantity() );
        $line_total = (float) $item->get_total();
        $amount     = $line_total / $qty;
        if ( $amount <= 0 ) return;

        // Charge this unit only
        $intent_id = $gateway->charge_recurring_item_unit( $order_id, $item_id, $unit, $amount, $order->get_currency() );
        if ( $intent_id === 'skipped' ) return;

        // Reschedule next cycle for this unit
        $interval = wc_get_order_item_meta( $item_id, '_easy_subscriptions_period_interval', true )
                 ?: get_post_meta( $pid, '_easy_subscriptions_period_interval', true );
        $unit_str = wc_get_order_item_meta( $item_id, '_easy_subscriptions_period_unit', true )
                 ?: get_post_meta( $pid, '_easy_subscriptions_period_unit', true );
        if ( ! $interval || ! $unit_str ) return;

        $prev_next = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, true );
        $base_time = $prev_next > 0 ? $prev_next : time();
        $next_time = strtotime( "+{$interval} {$unit_str}", $base_time );
        if ( $next_time < time() + HOUR_IN_SECONDS ) {
            $next_time = time() + HOUR_IN_SECONDS;
        }

        wc_update_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, $next_time );

        as_unschedule_all_actions(
            'easy_subscriptions_process_payment',
            [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ]
        );

        as_schedule_single_action(
            $next_time,
            'easy_subscriptions_process_payment',
            [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ],
            'easy_subscriptions'
        );
    } finally {
        delete_post_meta( $order_id, $handler_lock_key );
    }
}, 10, 3 );