<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}
// Adjust trial product price in the cart
add_action( 'woocommerce_before_calculate_totals', function( $cart ) {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) ) return;

    foreach ( $cart->get_cart() as $cart_item ) {
        $product_id   = $cart_item['product_id'];
        $trial_length = (int) get_post_meta( $product_id, '_easy_subscriptions_trial_length', true );
        $trial_unit   = get_post_meta( $product_id, '_easy_subscriptions_trial_unit', true );

        if ( $trial_length > 0 && $trial_unit ) {
            if ( empty( $cart_item['original_price'] ) ) {
                $cart_item['original_price'] = $cart_item['data']->get_regular_price();
            }

            $cart_item['data']->set_price( 1 );
        }
    }
});

add_filter( 'woocommerce_get_price_html', function( $price_html, $product ) {
    $pid = $product->get_id();

    $trial_length  = get_post_meta( $pid, '_easy_subscriptions_trial_length', true );
    $trial_unit    = get_post_meta( $pid, '_easy_subscriptions_trial_unit', true );
    $period_length = get_post_meta( $pid, '_easy_subscriptions_period_interval', true );
    $period_unit   = get_post_meta( $pid, '_easy_subscriptions_period_unit', true );

    if ( $trial_length && $trial_unit && $period_length && $period_unit ) {
        $display_price = wc_price( $product->get_price() );

        // If interval is 1, don’t show the "1"
        if ( (int) $period_length === 1 ) {
            // translators: %s is the billing period unit (e.g. day, week, month, year).
            $interval_text = sprintf( '%s', $period_unit );
        } else {
            // translators: %1$d is the billing period length, %2$s is the billing period unit (e.g. day, week, month, year).
            $interval_text = sprintf( __( '%1$d %2$s', 'easy-subscriptions' ), $period_length, $period_unit );
        }
        // translators: 1: trial length, 2: trial unit (day, week, month, year), 3: display price (e.g. $9.99), 4: billing interval (e.g. "1 month" or "3 weeks").
        $subscription_text = __( 'Free %1$d-%2$s trial, then %3$s / %4$s', 'easy-subscriptions' );

        $price_html = sprintf(
            $subscription_text,
            $trial_length,
            esc_html( $trial_unit ),
            $display_price,
            esc_html( $interval_text )
        );
    }

    return $price_html;
}, 20, 2 );

add_filter( 'woocommerce_product_single_add_to_cart_text', 'easy_subs_trial_button_text', 20, 1 );
add_filter( 'woocommerce_product_add_to_cart_text', 'easy_subs_trial_button_text', 20, 1 );

function easy_subs_trial_button_text( $text ) {
    global $product;
    if ( ! $product ) return $text;

    $pid = $product->get_id();

    $trial_length = get_post_meta( $pid, '_easy_subscriptions_trial_length', true );
    $trial_unit   = get_post_meta( $pid, '_easy_subscriptions_trial_unit', true );

    if ( $trial_length && $trial_unit ) {
        return __( 'Start Free Trial', 'easy-subscriptions' );
    }

    return $text;
}

/**
 * Show trial charge/refund notice at top of cart & checkout
 */



// Add the subscription extra fee to the order if set
add_action( 'woocommerce_cart_calculate_fees', 'easy_subscriptions_add_additional_fee', 20, 1 );
function easy_subscriptions_add_additional_fee( $cart ) {
    if ( is_admin() && ! defined( 'DOING_AJAX' ) ) {
        return;
    }

    $extra_fee_total = 0;

    foreach ( $cart->get_cart() as $cart_item ) {
        $product_id = $cart_item['product_id'];

        // Handle variations properly
        if ( isset( $cart_item['variation_id'] ) && $cart_item['variation_id'] > 0 ) {
            $product_id = $cart_item['variation_id'];
        }

        $additional_fee = get_post_meta( $product_id, '_easy_subscriptions_additional_fee', true );

        if ( ! empty( $additional_fee ) && $additional_fee > 0 ) {
            // Only add once per product, not per quantity
            $extra_fee_total += floatval( $additional_fee );
        }
    }

    if ( $extra_fee_total > 0 ) {
        $cart->add_fee( __( 'Fees', 'easy-subscriptions' ), $extra_fee_total, false );
    }
}



// Offer one-time payment option
add_action( 'woocommerce_before_add_to_cart_button', function() {
    global $product;
    if ( ! $product ) {
        return;
    }

    $pid = $product->get_id();

    // Check if subscription is enabled
    $is_sub = get_post_meta( $pid, '_easy_subscriptions', true );
    if ( $is_sub !== 'yes' ) {
        return;
    }

    // Check if "offer one-time payment option" is enabled
    $offer_one_time = get_post_meta( $pid, '_easy_subscriptions_one_time', true );
    if ( $offer_one_time !== 'yes' ) {
        return;
    }

    ?>
    <div class="easy-subscriptions-choice" style="margin-bottom:15px;">
        <label>
            <input type="radio" name="easy_subscription_choice" value="subscription" checked="checked" />
            <?php esc_html_e( 'Subscribe', 'easy-subscriptions' ); ?>
        </label>
        <br/>
        <label>
            <input type="radio" name="easy_subscription_choice" value="one_time" />
            <?php esc_html_e( 'One-time purchase', 'easy-subscriptions' ); ?>
        </label>
        <?php wp_nonce_field( 'easy_subs_add_to_cart', 'easy_subs_nonce' ); ?>
    </div>
    <?php
});


// Show in cart/checkout (one-time payment option)
add_filter( 'woocommerce_get_item_data', function( $item_data, $cart_item ) {
    if ( isset( $cart_item['easy_subscription_choice'] ) ) {
        $choice = $cart_item['easy_subscription_choice'] === 'one_time'
            ? __( 'One-time purchase', 'easy-subscriptions' )
            : __( 'Subscription', 'easy-subscriptions' );
        $item_data[] = [
            'name'  => __( 'Purchase Option', 'easy-subscriptions' ),
            'value' => $choice,
        ];
    }
    return $item_data;
}, 10, 2 );

// Save to order item meta (one-time payment option)
add_action( 'woocommerce_add_order_item_meta', function( $item_id, $values ) {
    if ( isset( $values['easy_subscription_choice'] ) ) {
        wc_add_order_item_meta( $item_id, '_easy_subscription_choice', $values['easy_subscription_choice'] );
    }
}, 10, 2 );


add_filter( 'woocommerce_add_cart_item_data', function( $cart_item_data, $product_id, $variation_id ) {
    if (
        isset( $_POST['easy_subscription_choice'], $_POST['easy_subs_nonce'] )
        && wp_verify_nonce(
            sanitize_text_field( wp_unslash( $_POST['easy_subs_nonce'] ) ),
            'easy_subs_add_to_cart'
        )
    ) {
        $cart_item_data['easy_subscription_choice'] = sanitize_text_field( wp_unslash( $_POST['easy_subscription_choice'] ) );
    }

    return $cart_item_data;
}, 10, 3 );



//shipping cycle
/**
 * Schedule shipping cycle orders
 */
add_action( 'woocommerce_thankyou', function( $order_id ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;

    foreach ( $order->get_items() as $item_id => $item ) {
        $pid = $item->get_variation_id() ?: $item->get_product_id();
        $separate_shipping = get_post_meta( $pid, '_easy_subscriptions_separate_shipping', true );
        if ( $separate_shipping !== 'yes' ) continue;

        $shipping_interval = get_post_meta( $pid, '_easy_subscriptions_shipping_interval', true );
        $shipping_unit     = get_post_meta( $pid, '_easy_subscriptions_shipping_unit', true );
        $shipping_sync     = get_post_meta( $pid, '_easy_subscriptions_shipping_sync', true );

        $qty = (int) $item->get_quantity(); // 🔑 how many units we need to schedule

        for ( $unit = 1; $unit <= $qty; $unit++ ) {
            // Default base time = now
            $base_time = time();

            // 🔑 Calculate first shipping time
            if ( $shipping_sync && $shipping_sync !== 'off' ) {
                switch ( $shipping_sync ) {
                    case 'monday':
                        $shipping_time = strtotime( 'next monday 00:00:00', $base_time );
                        break;
                    case 'month':
                        $shipping_time = strtotime( 'first day of next month 00:00:00', $base_time );
                        break;
                    case 'year':
                        $shipping_time = strtotime( 'first day of january next year 00:00:00', $base_time );
                        break;
                    default:
                        $shipping_time = strtotime( "+{$shipping_interval} {$shipping_unit}", $base_time );
                }
            } else {
                $shipping_time = strtotime( "+{$shipping_interval} {$shipping_unit}", $base_time );
            }

            // Unschedule any existing cycles for this unit
            if ( function_exists( 'as_unschedule_all_actions' ) ) {
                as_unschedule_all_actions(
                    'easy_subscriptions_trigger_shipping_cycle',
                    [
                        'order_id' => (int) $order_id,
                        'item_id'  => (int) $item_id,
                        'unit'     => (int) $unit,
                    ]
                );
            }

            // Schedule first cycle for this unit
            as_schedule_single_action(
                $shipping_time,
                'easy_subscriptions_trigger_shipping_cycle',
                [
                    'order_id' => (int) $order_id,
                    'item_id'  => (int) $item_id,
                    'unit'     => (int) $unit,
                ],
                'easy-subscriptions'
            );

            //error_log( "EasySubs: scheduled first shipping cycle for order {$order_id}, item {$item_id}, unit {$unit} at " . gmdate( 'c', $shipping_time ) );
        }
    }
});


/**
 * Handle shipping cycle trigger
 */
/**
 * Handle shipping cycle trigger
 */
/**
 * Handle shipping cycle trigger
 */
add_action( 'easy_subscriptions_trigger_shipping_cycle', function( $order_id, $item_id, $unit ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;

    $item = $order->get_item( $item_id );
    if ( ! $item ) return;

    $status_key = '_easy_subscriptions_status_unit_' . (int) $unit;
    $unit_status = wc_get_order_item_meta( $item_id, $status_key, true );
//error_log( "EasySubs: unit status: {$unit}, item {$item_id}, statuskey: {$status_key} (status=cancelled)" );
    if ( $unit_status === 'cancelled' ) {
        //error_log( "EasySubs: Skipping shipping cycle for order {$order_id}, item {$item_id}, unit {$unit} (status=cancelled)" );
        return; // 🚫 Do not create order or schedule next cycle
    }

    $pid = $item->get_variation_id() ?: $item->get_product_id();

    $shipping_interval = get_post_meta( $pid, '_easy_subscriptions_shipping_interval', true );
    $shipping_unit     = get_post_meta( $pid, '_easy_subscriptions_shipping_unit', true );
    $shipping_sync     = get_post_meta( $pid, '_easy_subscriptions_shipping_sync', true );

    // --- Create new shipping cycle order ---
    $shipping_order = wc_create_order( [ 'status' => 'processing' ] );
    $new_order_number = $shipping_order->get_order_number();

    // ✅ Assign customer & addresses
    if ( $order->get_customer_id() ) {
        $shipping_order->set_customer_id( $order->get_customer_id() );
    }
    $shipping_order->set_address( $order->get_address( 'billing' ), 'billing' );
    $shipping_order->set_address( $order->get_address( 'shipping' ), 'shipping' );
    $shipping_order->set_billing_email( $order->get_billing_email() );
    $shipping_order->set_billing_phone( $order->get_billing_phone() );

    // ✅ Only add THIS item
    $product = $item->get_product();
    if ( $product ) {
        $shipping_order->add_product(
            $product,
            $item->get_quantity(),
            [
                'name'   => $product->get_name() . " (Order #{$new_order_number} Shipping cycle)",
                'totals' => [
                    'subtotal' => 0,
                    'total'    => 0,
                ],
            ]
        );
    }

    // ✅ Mark as shipping cycle
    $shipping_order->update_meta_data( '_easy_shipping_cycle', 'yes' );
    $shipping_order->update_meta_data( '_easy_shipping_parent', $order_id );
    $shipping_order->update_meta_data( '_easy_shipping_item', $item_id );
    $shipping_order->save();

    // ✅ Add note to original order
    $order->add_order_note( sprintf(
        'Shipping cycle triggered for item "%s" on %s (Order #%s).',
        $item->get_name(),
        wc_format_datetime( new WC_DateTime() ),
        $shipping_order->get_order_number()
    ));
    $order->save();

    // --- Schedule next cycle ---
    $base_time = time();

    if ( $shipping_sync && $shipping_sync !== 'off' ) {
        switch ( $shipping_sync ) {
            case 'monday':
                $next_time = strtotime( 'next monday 00:00:00', $base_time );
                break;
            case 'month':
                $next_time = strtotime( 'first day of next month 00:00:00', $base_time );
                break;
            case 'year':
                $next_time = strtotime( 'first day of january next year 00:00:00', $base_time );
                break;
            default:
                $next_time = strtotime( "+{$shipping_interval} {$shipping_unit}", $base_time );
        }
    } else {
        $next_time = strtotime( "+{$shipping_interval} {$shipping_unit}", $base_time );
    }

as_schedule_single_action(
    $next_time,
    'easy_subscriptions_trigger_shipping_cycle',
    [
        'order_id' => (int) $order_id,
        'item_id'  => (int) $item_id,
        'unit'     => (int) $unit,
    ],
    'easy-subscriptions'
);

    //error_log( "EasySubs: scheduled next shipping cycle for order {$order_id}, item {$item_id} at " . gmdate( 'c', $next_time ) );
}, 10, 3 );


/**
 * Suppress emails for shipping cycle orders
 */
add_filter( 'woocommerce_email_enabled_new_order', function( $enabled, $order ) {
    if ( $order && $order->get_meta( '_easy_shipping_cycle' ) === 'yes' ) {
        return false;
    }
    return $enabled;
}, 10, 2 );

add_filter( 'woocommerce_email_enabled_customer_completed_order', function( $enabled, $order ) {
    if ( $order && $order->get_meta( '_easy_shipping_cycle' ) === 'yes' ) {
        return false;
    }
    return $enabled;
}, 10, 2 );

//exclude shipping status from reports
add_filter( 'woocommerce_analytics_exclude_order', function( $exclude, $order_id ) {
    $order = wc_get_order( $order_id );
    if ( $order && $order->get_meta( '_easy_shipping_cycle' ) === 'yes' ) {
        return true; // exclude from analytics
    }
    return $exclude;
}, 10, 2 );
