<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Easy_Subscriptions_Payment_Update {
    public function __construct() {
        // Capture target subscription when user lands on Add Payment Method
        add_action( 'template_redirect', [ $this, 'capture_subscription_target' ] );

        // Attach new tokens
        add_action( 'woocommerce_payment_token_created', [ $this, 'maybe_attach_token' ], 10, 2 );
        add_action( 'woocommerce_payment_token_set_default', [ $this, 'maybe_attach_default_token' ], 10, 2 );

        // Redirect back with a notice
        add_action( 'template_redirect', [ $this, 'maybe_redirect_back' ] );
    }

    /**
     * True when we should NOT run any of this logic.
     */
    private function should_skip_for_checkout(): bool {
        // WooCommerce not available yet
        if ( ! function_exists( 'is_checkout' ) ) {
            return false;
        }

        // Skip on checkout page + all checkout endpoints (order-pay, order-received, etc.)
        if ( is_checkout() ) {
            return true;
        }

        // Skip on checkout AJAX (update_order_review, wc-ajax endpoints, fragments, etc.)
        if ( function_exists( 'wp_doing_ajax' ) && wp_doing_ajax() ) {
            // If you only want to skip *WooCommerce* AJAX, keep it simple and skip all ajax:
            return true;
        }

        // Extra safety: some themes/plugins route checkout via ?wc-ajax=
        if ( function_exists( 'wp_doing_ajax' ) && wp_doing_ajax() ) {
            return true;
        }

        return false;
    }

    public function capture_subscription_target() {
        if ( is_account_page() && function_exists( 'is_wc_endpoint_url' ) && is_wc_endpoint_url( 'add-payment-method' ) ) {
            if (
                isset( $_GET['es_sub'], $_GET['_es_nonce'] ) &&
                wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_es_nonce'] ) ), 'easy_subs_capture' )
            ) {
                if ( null === WC()->session ) {
                    WC()->initialize_session();
                }

                WC()->session->set( 'easy_subs_target_subscription', absint( $_GET['es_sub'] ) );

                $return = isset( $_GET['es_return'] )
                    ? esc_url_raw( wp_unslash( $_GET['es_return'] ) )
                    : wc_get_account_endpoint_url( 'easy-subscriptions' );

                WC()->session->set( 'easy_subs_return_url', $return );
            }
        }
    }

private function attach_token_to_subscription( $token_id ) {
    $sub_id = WC()->session ? WC()->session->get( 'easy_subs_target_subscription' ) : 0;
    if ( ! $sub_id ) {
        return;
    }

    $token = WC_Payment_Tokens::get( $token_id );
    if ( ! $token || (int) $token->get_user_id() !== (int) get_current_user_id() ) {
        return;
    }

    $gateway = $token->get_gateway_id();
    $customer_id = $token->get_meta( 'customer_id' );

switch ( $gateway ) {
    case 'woocommerce_payments':
        update_post_meta( $sub_id, '_wcpay_payment_method_id', (int) $token->get_id() );
        update_post_meta( $sub_id, '_wcpay_payment_method_title', $token->get_display_name() );
        if ( $customer_id ) {
            update_post_meta( $sub_id, '_wcpay_customer_id', $customer_id );
        }
        break;

    case 'stripe':
        update_post_meta( $sub_id, '_stripe_payment_method_id', (int) $token->get_id() );
        update_post_meta( $sub_id, '_stripe_payment_method_title', $token->get_display_name() );
        if ( $customer_id ) {
            update_post_meta( $sub_id, '_stripe_customer_id', $customer_id );
        }
        break;

    case 'paypal':
        update_post_meta( $sub_id, '_paypal_payment_method_id', (int) $token->get_id() );
        update_post_meta( $sub_id, '_paypal_payment_method_title', $token->get_display_name() );
        if ( $customer_id ) {
            update_post_meta( $sub_id, '_paypal_billing_agreement_id', $customer_id );
        }
        break;

    default:
        update_post_meta( $sub_id, '_easy_subscriptions_payment_method_id', (int) $token->get_id() );
        update_post_meta( $sub_id, '_easy_subscriptions_payment_method_title', $token->get_display_name() );
        if ( $customer_id ) {
            update_post_meta( $sub_id, '_easy_subscriptions_customer_id', $customer_id );
        }
        break;
}

    if ( null === WC()->session ) {
        WC()->initialize_session();
    }
    WC()->session->set( 'easy_subs_updated', 1 );

   /** error_log( sprintf(
        'EasySubs: attached token_id=%d (gateway=%s, customer_id=%s) to subscription %d',
        (int) $token->get_id(),
        $gateway,
        $customer_id ?: 'none',
        $sub_id
    ) ); */
}

    public function maybe_attach_token( $token_id, $user_id ) {
        if ( $this->should_skip_for_checkout() ) {
            return;
        }
        if ( (int) $user_id === (int) get_current_user_id() ) {
            $this->attach_token_to_subscription( $token_id );
        }
    }

    public function maybe_attach_default_token( $token_id, $user_id ) {
        if ( $this->should_skip_for_checkout() ) {
            return;
        }
        if ( (int) $user_id === (int) get_current_user_id() ) {
            $this->attach_token_to_subscription( $token_id );
        }
    }

    public function maybe_redirect_back() {
        if ( ! is_account_page() ) return;

        $updated = WC()->session ? WC()->session->get( 'easy_subs_updated' ) : 0;
        if ( $updated ) {
            $return = WC()->session->get( 'easy_subs_return_url' ) ?: wc_get_account_endpoint_url( 'easy-subscriptions' );

            WC()->session->__unset( 'easy_subs_updated' );
            WC()->session->__unset( 'easy_subs_target_subscription' );
            WC()->session->__unset( 'easy_subs_return_url' );

            wc_add_notice( __( 'Payment method updated for your subscription.', 'easy-subscriptions' ), 'success' );
            wp_safe_redirect( $return );
            exit;
        }
    }
}

new Easy_Subscriptions_Payment_Update();