<?php
// Exit if accessed directly
if ( ! defined( 'ABSPATH' ) ) exit;

/** 
 * Use the correct filter for the top bar checkboxes,
 * register the conditional tab, and save the value.
 */
add_action( 'woocommerce_product_data_panels', function() {
    echo '<div id="easy_subscriptions_ps_setup" class="panel woocommerce_options_panel hidden">';
    do_action( 'easy_subscriptions_panel_content' );
    echo '</div>';
});
add_action( 'woocommerce_admin_process_product_object', 'easy_subscriptions_save_checkbox_value' );

add_filter('allowed_redirect_hosts', function($hosts) {
    $hosts[] = 'www.paypal.com';
    $hosts[] = 'paypal.com';
    // Add sandbox if needed
    $hosts[] = 'www.sandbox.paypal.com';
    return $hosts;
});



//thank you hook stripe and woo payments
add_action( 'woocommerce_thankyou', function( $order_id ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;


    foreach ( $order->get_items() as $item_id => $item ) {

        //check for one time payment
        $choice = wc_get_order_item_meta( $item_id, '_easy_subscription_choice', true );
        if ( $choice === 'one_time' ) {
            //error_log("EasySubs: one-time purchase, skipping subscription scheduling for order {$order_id} item {$item_id}");
            continue;
        }

        $pid = $item->get_variation_id() ?: $item->get_product_id();
        $is_sub = get_post_meta( $pid, '_easy_subscriptions', true );
        if ( $is_sub !== 'yes' ) continue;

        $product = $item->get_product();

        $interval = get_post_meta( $pid, '_easy_subscriptions_period_interval', true );
        $unit     = get_post_meta( $pid, '_easy_subscriptions_period_unit', true );
        if ( ! $interval || ! $unit ) continue;

        $time = strtotime("+{$interval} {$unit}");

        // Save cadence at item level
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_period_interval', $interval );
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_period_unit', $unit );
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_next_payment', $time );
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_status', 'active' );
        // Also store next payment and status per unit (optional but recommended)

        $order_obj = wc_get_order( $order_id );
        if ( $order_obj ) {
            $order_obj->update_meta_data( '_easy_subscriptions_next_payment', $time );
            $order_obj->update_meta_data( '_easy_subscriptions_scheduled', 'yes' );
            $order_obj->update_meta_data( '_easy_subscriptions_period_interval', $interval );
            $order_obj->update_meta_data( '_easy_subscriptions_period_unit', $unit );
            $order_obj->save();
        }
        
        $original_price = $product->get_regular_price();
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_original_price', $original_price );
        //error_log( "EasySubs: saved original price={$original_price} for order item {$item_id} (order {$order_id})" );

        $pm_title = $order->get_payment_method_title();  // e.g. 'Visa ending in 4242'
            if ( $pm_title ) {
                update_post_meta( $order_id, '_stripe_payment_method_title', $pm_title );
            }

        $qty = (int) $item->get_quantity();

        // Cancel existing actions for this item (all quantities)
        if ( function_exists( 'as_unschedule_all_actions' ) ) {
            as_unschedule_all_actions(
                'easy_subscriptions_process_payment',
                [ 'order_id' => (int) $order_id, 'item_id' => (int) $item_id ]
            );
        }
        $ends     = get_post_meta( $pid, '_easy_subscriptions_ends', true ); // 'never' or 'date'
        $end_date = get_post_meta( $pid, '_easy_subscriptions_end_date', true );
        $end_ts   = $end_date ? strtotime( $end_date . ' 23:59:59' ) : 0;

        // Schedule one action for each unit in the quantity
        for ( $i = 1; $i <= $qty; $i++ ) {

            wc_update_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $i, $time );
            wc_update_order_item_meta( $item_id, '_easy_subscriptions_status_u' . $i, 'active' );

            $trial_length = (int) get_post_meta( $pid, '_easy_subscriptions_trial_length', true );
            $trial_unit   = get_post_meta( $pid, '_easy_subscriptions_trial_unit', true );

            if ( $trial_length > 0 && $trial_unit ) {
                $time = strtotime("+{$trial_length} {$trial_unit}");
            } else {
                $time = strtotime("+{$interval} {$unit}");
            }

            // Refund trial charge after 10 minutes
            if ( $trial_length > 0 && $trial_unit ) {
                as_schedule_single_action(
                    time() + (10 * MINUTE_IN_SECONDS),
                    'easy_subscriptions_refund_trial',
                    [ (int) $order_id, (int) $item_id, (int) $i ] // indexed array
                );
            }

            if ( ($ends === 'never' || ! $end_ts || $time <= $end_ts )&& !in_array( $order->get_payment_method(), ['paypal', 'ppcp-gateway', 'paypal_standard', 'ppec_paypal'], true ) ) {
                as_schedule_single_action(
                    $time,
                    'easy_subscriptions_process_payment',
                    [ 
                        'order_id' => (int) $order_id, 
                        'item_id'  => (int) $item_id, 
                        'unit'     => $i // unique per unit
                    ],
                    'easy-subscriptions' // optional group
                );
                //error_log( "EasySubs: scheduled initial payment for order {$order_id}, item {$item_id}, unit {$i} at " . gmdate('c', $time) );
            } else {
                //error_log( "EasySubs: NOT scheduling payment for order {$order_id}, item {$item_id}, unit {$i} because end_date={$end_date}" );
            }
        }
    }
});

//add stripe customer id to woocommerce oder
add_action( 'woocommerce_thankyou', 'easy_subscriptions_save_stripe_ids', 30 );
function easy_subscriptions_save_stripe_ids( $order_id ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;

    $payment_method = $order->get_payment_method();
    if ( ! in_array( $payment_method, [ 'stripe', 'stripe_cc', 'woocommerce_payments' ], true ) ) {
        return;
    }

    $user_id = $order->get_user_id();

    // Try order meta first
    $customer_id = get_post_meta( $order_id, '_stripe_customer_id', true )
                ?: get_post_meta( $order_id, '_wc_stripe_customer_id', true )
                ?: get_post_meta( $order_id, '_wcpay_customer_id', true );

    $pm_id = get_post_meta( $order_id, '_stripe_payment_method_id', true )
          ?: get_post_meta( $order_id, '_wcpay_payment_method_id', true );

    // Fallback: grab from tokens
    if ( ( empty( $customer_id ) || empty( $pm_id ) ) && $user_id && class_exists( 'WC_Payment_Tokens' ) ) {
        $tokens = WC_Payment_Tokens::get_customer_tokens( $user_id, $payment_method );
        if ( ! empty( $tokens ) ) {
            $token = reset( $tokens );
            if ( $token ) {
                if ( empty( $customer_id ) ) {
                    $maybe = $token->get_meta( 'customer_id' );
                    if ( $maybe ) $customer_id = $maybe;
                }
                if ( empty( $pm_id ) && $token->get_token() ) {
                    $pm_id = $token->get_token();
                }
            }
        }
    }

    // Fallback: grab from usermeta (including prefixed key)
    if ( empty( $customer_id ) && $user_id ) {
        foreach ( [
            '_stripe_customer_id',
            '_wc_stripe_customer_id',
            '_wcpay_customer_id',
            'wp__stripe_customer_id' // ✅ your setup
        ] as $meta_key ) {
            $maybe = get_user_meta( $user_id, $meta_key, true );
            if ( ! empty( $maybe ) ) {
                $customer_id = $maybe;
                break;
            }
        }
    }

    // Save to order
    if ( ! empty( $customer_id ) ) {
        update_post_meta( $order_id, '_stripe_customer_id', $customer_id );
        update_post_meta( $order_id, '_wcpay_customer_id', $customer_id );
    }
    if ( ! empty( $pm_id ) ) {
        update_post_meta( $order_id, '_stripe_payment_method_id', $pm_id );
        update_post_meta( $order_id, '_wcpay_payment_method_id', $pm_id );
    }

    //error_log( "EasySubs: persisted Stripe IDs for order {$order_id} (customer_id={$customer_id}, pm_id={$pm_id})" );
}



//refund the free trial inital deduction for stripe
add_action( 'easy_subscriptions_refund_trial', function( $order_id, $item_id, $unit ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;

    $refund_amount = 1.00;
    $currency      = $order->get_currency();

    // Detect gateway by order
    $gateway = wc_get_payment_gateway_by_order( $order );
    $gateway_id = $gateway ? $gateway->id : '';

    // --- Stripe / WooPayments flow ---
    if ( $gateway && method_exists( $gateway, 'process_refund' ) && in_array( $gateway_id, [ 'stripe', 'woocommerce_payments' ], true ) ) {
        $result = $gateway->process_refund(
            $order_id,
            $refund_amount,
            "Trial refund (unit {$unit})"
        );

        if ( is_wp_error( $result ) ) {
            //error_log( "EasySubs: Stripe refund failed for order {$order_id}: " . $result->get_error_message() );
            return;
        }

        $order->add_order_note( sprintf(
            'Trial refund of %s processed via Stripe/WooPayments for unit %d.',
            wc_price( $refund_amount ),
            $unit
        ));
        //error_log("EasySubs: Stripe/Woo refund success for order {$order_id}, unit {$unit}, amount={$refund_amount}");
        return;
    }

    // --- PayPal / PPCP flow ---
    $capture_id = $order->get_transaction_id();
    if ( $capture_id ) {
        $paypal_gateway = new Easy_Subscriptions_Gateway_PayPal();
        $result = $paypal_gateway->refund_capture(
            $capture_id,
            $refund_amount,
            $currency,
            "Trial refund (unit {$unit})"
        );

        if ( isset( $result['id'] ) ) {
            $order->add_order_note( sprintf(
                'Trial refund of %s processed via PayPal/PPCP for unit %d.',
                wc_price( $refund_amount ),
                $unit
            ));
            //error_log("EasySubs: PayPal refund success for order {$order_id}, unit {$unit}, capture={$capture_id}");
        } else {
            //error_log("EasySubs: PayPal refund failed for order {$order_id}, response=" . print_r( $result, true ));
        }
        return;
    }

    // --- No supported gateway ---
    //error_log("EasySubs: No supported gateway found for trial refund on order {$order_id}");

}, 10, 3 );


// recurring payment for woo payments and stripe
add_action( 'easy_subscriptions_process_payment', function( $order_id, $item_id, $unit ) {
    $order_id = (int) $order_id;
    $item_id  = (int) $item_id;
    $unit     = (int) $unit;

    $handler_lock_key = '_easy_subscriptions_handler_processing_' . $item_id . '_' . $unit;
    if ( ! add_post_meta( $order_id, $handler_lock_key, time(), true ) ) {
        //error_log( "EasySubs: handler lock exists, skipping for #{$order_id} item {$item_id} unit {$unit}" );
        return;
    }

    try {
        
        $order = wc_get_order( $order_id );
        if ( ! $order ) return;
        
        $payment_method = $order->get_payment_method();
        $gateway        = null;
        //error_log('Payment method is: ' . $payment_method);
        switch ( $payment_method ) {
            case 'woocommerce_payments':
                $gateway = new Easy_Subscriptions_Gateway_WooPayments();
                break;

            case 'stripe':
            case 'stripe_cc': // sometimes WooCommerce Stripe extension uses this
                $gateway = new Easy_Subscriptions_Gateway_Stripe();
                break;

            default:
                // fallback: Stripe, or you can throw an error
                $gateway = new Easy_Subscriptions_Gateway_Stripe();
                break;
        }

        if ( ! $gateway ) {
            //error_log( "EasySubs: No gateway available for payment method {$payment_method}" );
            return;
        }

        $order   = wc_get_order( $order_id );
        if ( ! $order ) return;

        $item = $order->get_item( $item_id );
        if ( ! $item ) return;

        $pid    = $item->get_variation_id() ?: $item->get_product_id();
        $is_sub = get_post_meta( $pid, '_easy_subscriptions', true );
        if ( $is_sub !== 'yes' ) return;

        $qty = max( 1, (int) $item->get_quantity() );

        $original_price = wc_get_order_item_meta( $item_id, '_easy_subscriptions_original_price', true );
        $amount         = $original_price;

        if ( $amount <= 0 ) return;

        // ---- Charge this unit ----
        try {
        $intent_id = $gateway->charge_recurring_item_unit(
            $order_id,
            $item_id,
            $unit,
            $amount,
            $order->get_currency()
        );}catch ( \Exception $e ) {
            easy_subscriptions_handle_failed_attempt( $order_id, $item_id,$unit );
            return;
        }

        if ( $intent_id === 'skipped' ) {
            //error_log( "EasySubs: skipped duplicate charge for #{$order_id} item {$item_id} unit {$unit}, rescheduling next cycle." );
        } else {
            delete_post_meta( $order_id, '_easy_subscriptions_failed_attempts' );
            update_post_meta( $order_id, '_easy_subscriptions_status', 'active' );
            wc_update_order_item_meta(
                    $item_id,
                    '_easy_subscriptions_status_unit_' . $unit,
                    'active'
            );
        }

        // ---- Schedule Next Cycle ----
        $interval = wc_get_order_item_meta( $item_id, '_easy_subscriptions_period_interval', true )
                 ?: get_post_meta( $pid, '_easy_subscriptions_period_interval', true );
        $unit_str = wc_get_order_item_meta( $item_id, '_easy_subscriptions_period_unit', true )
                 ?: get_post_meta( $pid, '_easy_subscriptions_period_unit', true );

        if ( ! $interval || ! $unit_str ) return;

        $prev_next = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, true );
        if ( $prev_next <= 0 ) {
            $prev_next = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment', true );
        }
        if ( $prev_next <= 0 ) {
            $prev_next = (int) get_post_meta( $order_id, '_easy_subscriptions_next_payment', true );
        }

        $base_time = $prev_next > 0 ? $prev_next : time();
        $next_time = strtotime( "+{$interval} {$unit_str}", $base_time );

        if ( $next_time < time() + HOUR_IN_SECONDS ) {
            $next_time = time() + HOUR_IN_SECONDS;
        }

        wc_update_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, $next_time );
        wc_update_order_item_meta( $item_id, '_easy_subscriptions_next_payment', $next_time );
        update_post_meta( $order_id, '_easy_subscriptions_next_payment', $next_time );

        $ends     = get_post_meta( $pid, '_easy_subscriptions_ends', true ); 
        $end_date = get_post_meta( $pid, '_easy_subscriptions_end_date', true );
        $end_ts   = $end_date ? strtotime( $end_date . ' 23:59:59' ) : 0;

        // ✅ Block scheduling if beyond end date
        if ( !($ends === 'never') && ($end_ts && $next_time > $end_ts) ) {
            //error_log( "EasySubs: not scheduling further payments for #{$order_id} item {$item_id} unit {$unit}, end_date reached" );
            wc_update_order_item_meta( $item_id, '_easy_subscriptions_status_u' . $unit, 'ended' );
            wc_update_order_item_meta( $item_id, '_easy_subscriptions_status', 'ended' );
            return;
        }

        if ( function_exists( 'as_unschedule_all_actions' ) ) {
            as_unschedule_all_actions(
                'easy_subscriptions_process_payment',
                [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ]
            );
        }

        $already_next = as_next_scheduled_action(
            'easy_subscriptions_process_payment',
            [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ]
        );

        if ( $already_next && $already_next >= $next_time ) {
            $ts = gmdate( 'c', $already_next );
            //error_log( "EasySubs: next payment already scheduled for #{$order_id} item {$item_id} unit {$unit} at {$ts}" );
        } else {
            as_schedule_single_action(
                $next_time,
                'easy_subscriptions_process_payment',
                [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ],
                'easy-subscriptions'
            );
            //error_log( "EasySubs: scheduled next payment for #{$order_id} item {$item_id} unit {$unit} at " . gmdate( 'c', $next_time ) );
        }

        // Schedule next cycle
        as_schedule_single_action(
            $next_time,
            'easy_subscriptions_process_payment',
            [ 'order_id' => $order_id, 'item_id' => $item_id, 'unit' => $unit ],
            'easy-subscriptions'
        );

    } finally {
        delete_post_meta( $order_id, $handler_lock_key );
    }
}, 10, 3 );

//thank you hook for paypal to create a subscription ------------
add_action('woocommerce_thankyou', function($order_id) {
    $order = wc_get_order($order_id);
    if (!$order) return;

    // Only run for PayPal gateway
    if ($order->get_payment_method() !== 'ppcp-gateway') {
        return;
    }

    try {
        //Find subscription item (only one allowed with PayPal)
        $sub_item  = null;
        foreach ($order->get_items() as $item_id => $item) {
            $pid    = $item->get_variation_id() ?: $item->get_product_id();
            $is_sub = get_post_meta($pid, '_easy_subscriptions', true);
            if ($is_sub === 'yes') {
                $sub_item = $item;
                break;
            }
        }

        // Use the subscription item’s own price, not the order total
        $product = $sub_item->get_product();
        $amount  = (float) $product->get_regular_price();
        $currency = $order->get_currency();
        $customer_data = [
            'email' => $order->get_billing_email(),
        ];

        // Create the PayPal subscription (with trial → no double charge)
        $gateway = new Easy_Subscriptions_Gateway_PayPal();
        $sub_id  = $gateway->charge_initial($order_id, $amount, $currency, $customer_data);

        if ($sub_id) {
            update_post_meta($order_id, '_paypal_subscription_id', $sub_id);

            // Wait until PayPal tells us subscription is ACTIVE before marking paid
            add_action('init', function() use ($order_id, $sub_id, $gateway) {
                // phpcs:ignore WordPress.Security.NonceVerification.Recommended, WordPress.Security.NonceVerification.Missing
                if (isset($_GET['paypal-sub-return']) && $_GET['paypal-sub-return'] == $order_id) {
                    $token = $gateway->get_access_token();
                    $sub   = $gateway->api_request('GET', "v1/billing/subscriptions/{$sub_id}", [], $token);

                    $order = wc_get_order($order_id);
                    if (!$order) return;

                    if (!empty($sub['status']) && $sub['status'] === 'ACTIVE') {
                        if ($order->needs_payment()) {
                            $order->payment_complete($sub_id);
                        } else {
                            $order->update_status('processing', "PayPal subscription {$sub_id} activated.");
                        }
                        $order->add_order_note("PayPal subscription {$sub_id} is active.");
                    } else {
                        $order->update_status('on-hold', 'Awaiting PayPal subscription approval.');
                    }
                }
            });

            //error_log("EasySubs/PayPal: created subscription {$sub_id} for order {$order_id}, amount={$amount} {$currency}");
        }
    } catch (Exception $e) {
        //error_log("EasySubs/PayPal: failed to create subscription for order {$order_id} → " . $e->getMessage());
    }
}, 20, 1);



/**
 * Handle a failed recurring billing attempt: mark status, increment attempts, decide retry or final action.
 */
function easy_subscriptions_handle_failed_attempt( $order_id, $item_id, $unit ) {
    $order_id = (int) $order_id;
    $item_id  = (int) $item_id;
    $unit     = (int) $unit;
    $now      = time();

    // Settings
    $grace_hours   = (int) get_option( 'easy_subscriptions_payment_grace_hours', 48 );
    $status_action = get_option( 'easy_subscriptions_payment_status_action', 'suspended' ); // 'suspended' or 'cancelled'
    $grace_days    = (int) get_option( 'easy_subscriptions_payment_grace_days', 20 );
    $retry_enabled = (bool) get_option( 'easy_subscriptions_retry_failed_payments', false );

    // Track failed attempts at item+unit level
    $failed_key = "_easy_subscriptions_failed_attempts_u{$unit}";
    $failed_attempts = (int) wc_get_order_item_meta( $item_id, $failed_key, true );
    $failed_attempts = max( 0, $failed_attempts );

    // First failure → mark failed
    if ( $failed_attempts === 0 ) {
        wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'failed' );
    }

    // Increment attempts
    $failed_attempts++;
    wc_update_order_item_meta( $item_id, $failed_key, $failed_attempts );

    // Original due time
    $original_due = (int) wc_get_order_item_meta( $item_id, "_easy_subscriptions_next_payment_u{$unit}", true );
    if ( ! $original_due ) {
        $original_due = (int) get_post_meta( $order_id, '_easy_subscriptions_next_payment', true );
    }
    $elapsed = $original_due > 0 ? ( $now - $original_due ) : 0;

    // === Logic ===

    // Case 1: No grace
    if ( $grace_hours === 0 ) {
        as_unschedule_all_actions( 'easy_subscriptions_process_payment', [
            'order_id' => $order_id,
            'item_id'  => $item_id,
            'unit'     => $unit,
        ] );

        if ( $status_action === 'cancelled' ) {
            wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'cancelled' );
        } else {
            wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'suspended' );
            $when = $now + max( 1, $grace_days ) * DAY_IN_SECONDS;
            as_schedule_single_action( $when, 'easy_subscriptions_force_cancel', [
                'order_id' => $order_id,
                'item_id'  => $item_id,
                'unit'     => $unit,
            ] );
        }
        //easy_subscriptions_update_order_summary_status( $order_id );
        return;
    }

    // Case 2: Grace > 0 and still within window
    if ( $original_due > 0 && $elapsed < ( $grace_hours * HOUR_IN_SECONDS ) ) {
        if ( $retry_enabled ) {
            as_unschedule_all_actions( 'easy_subscriptions_process_payment', [
                'order_id' => $order_id,
                'item_id'  => $item_id,
                'unit'     => $unit,
            ] );
            $retry_at = $now + 6 * HOUR_IN_SECONDS;
            as_schedule_single_action( $retry_at, 'easy_subscriptions_process_payment', [
                'order_id' => $order_id,
                'item_id'  => $item_id,
                'unit'     => $unit,
            ] );
        } else {
            wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", $status_action );
        }
        //easy_subscriptions_update_order_summary_status( $order_id );
        return;
    }

    // Case 3: Grace expired
    as_unschedule_all_actions( 'easy_subscriptions_process_payment', [
        'order_id' => $order_id,
        'item_id'  => $item_id,
        'unit'     => $unit,
    ] );
    if ( $status_action === 'cancelled' ) {
        wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'cancelled' );
    } else {
        wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'suspended' );
        $when = $now + max( 1, $grace_days ) * DAY_IN_SECONDS;
        as_schedule_single_action( $when, 'easy_subscriptions_force_cancel', [
            'order_id' => $order_id,
            'item_id'  => $item_id,
            'unit'     => $unit,
        ] );
    }
    //easy_subscriptions_update_order_summary_status( $order_id );
}

/**
 * Force cancel after suspension window, but only if still suspended.
 */
add_action( 'easy_subscriptions_force_cancel', function( $order_id, $item_id = 0, $unit = 0 ) {
    $status = wc_get_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", true );

    if ( $status === 'suspended' ) {
        wc_update_order_item_meta( $item_id, "_easy_subscriptions_status_unit_{$unit}", 'cancelled' );

        if ( function_exists( 'as_unschedule_all_actions' ) ) {
            as_unschedule_all_actions(
                'easy_subscriptions_process_payment',
                [ 'order_id' => (int) $order_id, 'item_id' => (int) $item_id, 'unit' => (int) $unit ]
            );
        }
    }
}, 10, 3 );

add_filter( 'woocommerce_hidden_order_itemmeta', function( $hidden ) {
    $hidden = array_merge( $hidden, [
        '_easy_subscriptions',
        '_easy_subscriptions_period_interval',
        '_easy_subscriptions_period_unit',
        '_easy_subscriptions_next_payment',
        '_easy_subscriptions_next_payment_u',
        '_easy_subscriptions_next_payment_u1',
        '_easy_subscriptions_next_payment_u2',
        '_easy_subscriptions_status_u',
        '_easy_subscriptions_current_cycle_u1',

    ] );
    return $hidden;
});

add_action( 'woocommerce_thankyou', function( $order_id ) {
    $order = wc_get_order( $order_id );
    if ( ! $order ) return;

    if ( $order->get_payment_method() !== 'woocommerce_payments' ) {
        return;
    }

    $user_id = $order->get_user_id();
    if ( ! $user_id ) {
        return;
    }

    $tokens = WC_Payment_Tokens::get_customer_tokens( $user_id, 'woocommerce_payments' );
    if ( empty( $tokens ) ) {
        //error_log( "EasySubs: no WCPay tokens found for user {$user_id} on order {$order_id}" );
        return;
    }

    $token = reset( $tokens );
    if ( ! $token ) {
        //error_log( "EasySubs: no valid token object for user {$user_id} on order {$order_id}" );
        return;
    }

    // ✅ Save numeric token ID, not object
    $pm_id       = (int) $token->get_id();
    $customer_id = $token->get_meta( 'customer_id' );

    update_post_meta( $order_id, '_wcpay_payment_method_id', $pm_id );
    if ( $customer_id ) {
        update_post_meta( $order_id, '_wcpay_customer_id', $customer_id );
    }

    //error_log( "EasySubs: saved WCPay token_id={$pm_id}, customer_id={$customer_id} for order {$order_id}" );
}, 20 );


add_action('template_redirect', function() {
    // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended
    if (!isset($_GET['paypal-sub-return'])) {
        return;
    }
    // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended
    $order_id = absint($_GET['paypal-sub-return']);
    // phpcs:ignore WordPress.Security.NonceVerification.Missing, WordPress.Security.NonceVerification.Recommended
    $sub_id = isset( $_GET['sub_id'] ) ? sanitize_text_field( wp_unslash( $_GET['sub_id'] ) ) : '';
    if (!$order_id || !$sub_id) {
        return;
    }

    $order = wc_get_order($order_id);
    if (!$order) {
        return;
    }

    $gateway = new Easy_Subscriptions_Gateway_PayPal();
    $token   = $gateway->get_access_token();
    $sub     = $gateway->api_request('GET', "v1/billing/subscriptions/{$sub_id}", [], $token);

    if (!empty($sub['status']) && $sub['status'] === 'ACTIVE') {
        // ✅ Mark WooCommerce order as paid
        if ($order->needs_payment()) {
            $order->payment_complete($sub_id);
        } else {
            $order->update_status('processing', "PayPal subscription {$sub_id} activated.");
        }

        // Redirect to clean thank you page
        wp_safe_redirect($order->get_checkout_order_received_url());
        exit;
    } else {
        // Subscription not active → keep it on-hold
        $order->update_status('on-hold', 'Awaiting PayPal subscription activation.');
    }
});


/**
 * Disable all PayPal payment methods during checkout processing
 */
add_filter('woocommerce_available_payment_gateways', 'easy_subscriptions_disable_paypal_gateways_checkout');
function easy_subscriptions_disable_paypal_gateways_checkout($available_gateways) {
    // Run only on checkout page (not order-pay or other endpoints)
    if ( is_checkout() && ! is_wc_endpoint_url() ) {
        
        // If there is more than 1 subscription in cart
        if ( function_exists('easy_subscriptions_count_in_cart') && easy_subscriptions_count_in_cart() > 1 ) {
            $disabled_gateways = array(
                'paypal',
                'ppcp-gateway',
                'paypal_standard', 
                'ppec_paypal'
            );
            
            foreach ( $disabled_gateways as $gateway_id ) {
                if ( isset($available_gateways[$gateway_id]) ) {
                    unset($available_gateways[$gateway_id]);
                }
            }
        }
    }
    
    return $available_gateways;
}





// 🔍 Helper: count subscription products in cart
function easy_subscriptions_count_in_cart() {
    $count = 0;

    if ( ! WC()->cart ) return 0;

    foreach ( WC()->cart->get_cart() as $cart_item ) {
        $product_id = $cart_item['variation_id'] ?: $cart_item['product_id'];
        $is_sub     = get_post_meta( $product_id, '_easy_subscriptions', true );
        if ( $is_sub === 'yes' ) {
            $count += (int) $cart_item['quantity'];
        }
    }

    return $count;
}




add_action( 'template_redirect', function() {
    if ( ! function_exists( 'is_cart' ) || ! function_exists( 'is_checkout' ) ) {
        return;
    }

    // Only fire on cart or checkout page
    if ( ! is_cart() && ! is_checkout() ) {
        return;
    }

    if ( ! WC()->cart ) {
        return;
    }

    $show_notice = false;
    $show_notice_default = false;

    foreach ( WC()->cart->get_cart() as $cart_item ) {
        $product_id = $cart_item['variation_id'] ?: $cart_item['product_id'];

        $is_sub       = get_post_meta( $product_id, '_easy_subscriptions', true );
        $trial_length = (int) get_post_meta( $product_id, '_easy_subscriptions_trial_length', true );
        $trial_unit   = get_post_meta( $product_id, '_easy_subscriptions_trial_unit', true );

        // Must be a subscription AND have a trial period
        if ( $is_sub === 'yes' && $trial_length > 0 && $trial_unit ) {
            $show_notice = true;
            break;
        } elseif ($is_sub === 'yes'){
            $show_notice_default = true;
        }
    }

    if ( $show_notice ) {
        wc_add_notice(
            sprintf(
                // translators: %s is the formatted trial charge amount (e.g. $1.00).
                __( 'We’ll place a temporary charge of %s on your card to confirm it for this subscription. Don’t worry — it will be refunded within 10 minutes.', 'easy-subscriptions' ),
                wc_price( 1 ) // hard-coded trial charge; replace with dynamic if needed
            ),
            'notice'
        );
    } elseif ( $show_notice_default ) {

        wc_add_notice(
            __( 'Subscription in cart. Your payment will be securely stored for recurring charges.', 'easy-subscriptions' ),
            'notice'
        );

        // --- PayPal-only error notice gate ---
        $paypal_active = false;

        if ( function_exists( 'WC' ) && WC()->payment_gateways() ) {
            // Ensure gateways are loaded
            $gateways = WC()->payment_gateways()->get_available_payment_gateways();

            // Common PayPal gateway IDs across WooCommerce/PayPal plugins
            $paypal_ids = array(
                'paypal',                // WooCommerce "PayPal Standard" (legacy)
                'ppec_paypal',           // PayPal Checkout (older)
                'ppcp-gateway',          // WooCommerce PayPal Payments (common)
                'ppcp',                  // some installs
                'payPal',                // just-in-case weird casing
            );

            foreach ( $paypal_ids as $id ) {
                if ( isset( $gateways[ $id ] ) ) {
                    $paypal_active = true;
                    break;
                }
            }

            // Extra fallback: sometimes a gateway is enabled but not "available" until checkout context.
            // So also check enabled gateways list.
            if ( ! $paypal_active ) {
                $all_gateways = WC()->payment_gateways()->payment_gateways();
                foreach ( $paypal_ids as $id ) {
                    if ( isset( $all_gateways[ $id ] ) && 'yes' === $all_gateways[ $id ]->enabled ) {
                        $paypal_active = true;
                        break;
                    }
                }
            }
        }

        if ( $paypal_active && function_exists( 'easy_subscriptions_count_in_cart' ) && easy_subscriptions_count_in_cart() > 1 ) {
            wc_add_notice(
                __( 'PayPal payment methods are unavailable when ordering multiple subscriptions.', 'easy-subscriptions' ),
                'error'
            );
        }
    }
}, 20 );

