<?php
if ( ! defined( 'ABSPATH' ) ) exit;

class Easy_Subscriptions_My_Account {

    public function __construct() {
        add_action( 'init', [ $this, 'add_endpoint' ] );
        add_filter( 'woocommerce_account_menu_items', [ $this, 'add_menu_item' ] );
        add_action( 'woocommerce_account_easy-subscriptions_endpoint', [ $this, 'render_endpoint' ] );
        add_action( 'template_redirect', [ $this, 'handle_cancel_request' ] );
    }

    public function add_endpoint() {
        add_rewrite_endpoint( 'easy-subscriptions', EP_ROOT | EP_PAGES );
    }

    public function add_menu_item( $items ) {
        $items['easy-subscriptions'] = __( 'My Subscriptions', 'easy-subscriptions' );
        return $items;
    }

    public function render_endpoint() {
        // Only include the template here (front-end endpoint render).
        $template = dirname( __DIR__ ) . '/templates/template-easy-subscriptions-my-account.php';
        if ( file_exists( $template ) ) {
            include $template;
        } else {
            echo '<p>' . esc_html__( 'Template not found.', 'easy-subscriptions' ) . '</p>';
        }
    }

public function handle_cancel_request() {
    if ( is_account_page() 
        && isset($_GET['action'], $_GET['order_id'], $_GET['item_id'], $_GET['unit']) 
        && $_GET['action'] === 'cancel_subscription' ) {

        $order_id = absint($_GET['order_id']);
        $item_id  = absint($_GET['item_id']);
        $unit     = absint($_GET['unit']);
        $order    = wc_get_order($order_id);
        $paypal_plan_id = get_post_meta( $order_id, '_paypal_subscription_id', true );

        // Nonce check
if ( empty( $_GET['_wpnonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_GET['_wpnonce'] ) ), 'cancel_subscription_' . $order_id . '_' . $item_id . '_' . $unit)) {
    wc_add_notice( __( 'Invalid action. Please try again.', 'easy-subscriptions' ), 'error' );
    wp_safe_redirect( wc_get_account_endpoint_url( 'easy-subscriptions' ) );
    exit;
}

        if (( $order && $order->get_user_id() === get_current_user_id() ) && empty( $paypal_plan_id )) {

            // Load restriction settings
            $time_after_creation = (int) get_option( 'easy_subscriptions_time_after_creation', 0 );
            $time_before_renew   = (int) get_option( 'easy_subscriptions_time_before_renew', 0 );

            $now         = current_time( 'timestamp' );
            $created_ts  = $order->get_date_created() ? $order->get_date_created()->getTimestamp() : 0;

            $allow_cancel = true;

            // Restrict if within "time after creation"
            if ( $time_after_creation > 0 && ( $now - $created_ts ) < ( $time_after_creation * DAY_IN_SECONDS ) ) {
                $allow_cancel = false;
            }

            // Restrict if within "time before renew date"
            if ( $time_before_renew > 0 && function_exists( 'as_next_scheduled_action' ) ) {
                $next_renew_ts = as_next_scheduled_action(
                    'easy_subscriptions_process_payment',
                    [
                        'order_id' => (int) $order_id,
                        'item_id'  => (int) $item_id,
                        'unit'     => (int) $unit,
                    ]
                );

                if ( $next_renew_ts && $next_renew_ts > 0 ) {
                    $diff = $next_renew_ts - $now;

                    if ( $diff > 0 && $diff < ( $time_before_renew * DAY_IN_SECONDS ) ) {
                        $allow_cancel = false;
                    }
                }
            }

            if ( ! $allow_cancel ) {
                wc_add_notice( __( 'This subscription cannot be cancelled at this time.', 'easy-subscriptions' ), 'error' );
                wp_safe_redirect( wc_get_account_endpoint_url( 'easy-subscriptions' ) );
                exit;
            }

            // Proceed with cancellation for this specific unit
            if ( function_exists('as_unschedule_all_actions') ) {
                as_unschedule_all_actions(
                    'easy_subscriptions_process_payment',
                    [
                        'order_id' => (int) $order_id,
                        'item_id'  => (int) $item_id,
                        'unit'     => (int) $unit,
                    ]
                );
            }

            wc_update_order_item_meta( $item_id, '_easy_subscriptions_status_unit_' . $unit, 'cancelled' );
            wc_delete_order_item_meta( $item_id, '_easy_subscriptions_next_payment_unit_' . $unit );

            wc_add_notice( __( 'Your subscription unit has been cancelled.', 'easy-subscriptions' ), 'success' );
            wp_safe_redirect( wc_get_account_endpoint_url( 'easy-subscriptions' ) );
            exit;
        } else if ( ! empty( $paypal_plan_id ) ) {
            // Make sure gateway class is loaded
            if ( class_exists( 'Easy_Subscriptions_Gateway_PayPal' ) ) {
                $gateway = new Easy_Subscriptions_Gateway_PayPal();

                // Retrieve the subscription ID you saved earlier
                $subscription_id = get_post_meta( $order_id, '_paypal_subscription_id', true );

                if ( ! empty( $subscription_id ) ) {
                    try {
                        $gateway->cancel_subscription( $subscription_id );
                        // Optionally: update order/subscription status
                        update_post_meta( $order_id, '_paypal_subscription_status', 'cancelled' );
                        wc_update_order_item_meta( $item_id, '_easy_subscriptions_status_unit_' . $unit, 'cancelled' );
                    } catch ( Exception $e ) {
                        //error_log( 'PayPal cancellation failed: ' . $e->getMessage() );
                    }
                }
            }
        } else {
            wc_add_notice( __( 'Invalid subscription cancel request.', 'easy-subscriptions' ), 'error' );
            wp_safe_redirect( wc_get_account_endpoint_url( 'easy-subscriptions' ) );
            exit;
        }
    }
}

/**
 * Determine subscription status for a specific unit of an order item.
 * Priority: scheduled action > per-unit meta > item meta > order meta > next-payment meta.
 *
 * @param int $order_id
 * @param int $item_id
 * @param int $unit
 * @return string 'active' | 'cancelled' | 'suspended' | 'failed'
 */
public static function unit_status_myaccount( $order_id, $item_id, $unit ) {
    $order_id = (int) $order_id;
    $item_id  = (int) $item_id;
    $unit     = (int) $unit;

    $paypal_plan_id = get_post_meta( $order_id, '_paypal_subscription_id', true );
    $paypal_sub_status = get_post_meta( $order_id, '_paypal_subscription_status', true );

    $status = wc_get_order_item_meta( $item_id, '_easy_subscriptions_status_unit_' . $unit, true );
    if ( ! empty( $paypal_plan_id ) && (strtolower( $paypal_sub_status ) !== 'cancelled' )) {
        return 'active';
    }

    // 1) Action Scheduler takes precedence: if no future job, it's cancelled.
    if ( function_exists( 'as_next_scheduled_action' ) ) {
        $next_ts = as_next_scheduled_action(
            'easy_subscriptions_process_payment',
            [
                'order_id' => $order_id,
                'item_id'  => $item_id,
                'unit'     => $unit,
            ]
        );
        if ( ! $next_ts || $next_ts <= time() ) {
            if ($status == 'suspended'){
                 return 'suspended';
            }
            return 'cancelled';
        }
    }

    // 2) Per-unit status meta
    if ( $status ) {
        return $status;
    }

    // 3) Item-level status meta
    $status = wc_get_order_item_meta( $item_id, '_easy_subscriptions_status', true );
    if ( $status ) {
        return $status;
    }

    // 4) Order-level status meta
    $status = get_post_meta( $order_id, '_easy_subscriptions_status', true );
    if ( $status ) {
        return $status;
    }

    // 5) Fallback: next payment meta (support both key styles you used)
    $next_meta = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, true );
    if ( ! $next_meta ) {
        $next_meta = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_unit_' . $unit, true );
    }
    if ( ! $next_meta ) {
        $next_meta = (int) wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment', true );
    }
    if ( $next_meta && $next_meta > time() ) {
        return 'active';
    }

    return 'cancelled';
}
}

new Easy_Subscriptions_My_Account();
if ( ! function_exists( 'easy_subs_unit_status_myaccount' ) ) {
    /**
     * Global wrapper so templates can call easy_subs_unit_status(...)
     */
    function easy_subs_unit_status_myaccount( $order_id, $item_id, $unit ) {
        return Easy_Subscriptions_My_Account::unit_status_myaccount( $order_id, $item_id, $unit );
    }
}

