<?php
if ( ! defined( 'ABSPATH' ) ) exit;
//force save payment method

/**
 * BLOCKS / Store API: Block checkout if a subscription is in the cart/order
 * unless "save payment method" is enabled (or a saved token is used).
 */
add_action( 'woocommerce_store_api_checkout_update_order_from_request', function( $order, $request ) {

	// 1) Only enforce when the ORDER contains a subscription item.
	$has_sub = false;
	foreach ( $order->get_items() as $item ) {
		$pid = $item->get_variation_id() ?: $item->get_product_id();
		if ( get_post_meta( $pid, '_easy_subscriptions', true ) === 'yes' ) {
			$has_sub = true;
			break;
		}
	}
	if ( ! $has_sub ) return;

	// 2) Build payment_data map (array of {key,value}).
	$payment_data = ( $request instanceof WP_REST_Request ) ? $request->get_param( 'payment_data' ) : [];
	if ( ! is_array( $payment_data ) ) $payment_data = [];

	$map = [];
	foreach ( $payment_data as $row ) {
		if ( is_array( $row ) && isset( $row['key'] ) ) {
			$map[ (string) $row['key'] ] = $row['value'] ?? null;
		}
	}

	// 3) Detect chosen gateway id (Blocks usually sends this as top-level param).
	$gateway_id = '';
	if ( $request instanceof WP_REST_Request ) {
		$gateway_id = (string) $request->get_param( 'payment_method' );
	}
	// Fallback if some gateway puts it into payment_data
	if ( ! $gateway_id && isset( $map['payment_method'] ) ) {
		$gateway_id = (string) $map['payment_method'];
	}
	if ( in_array( $gateway_id, [ 'ppcp-gateway', 'paypal', 'paypal_standard', 'ppec_paypal', 'ppcp' ], true ) ) {
		return;
	}

	// 4) Allow if customer chose an existing saved token (not "new").
	// Stripe/WCPay can differ, so accept ANY "*-payment-token" key.
	foreach ( $map as $k => $v ) {
		if ( is_string( $k ) && str_ends_with( $k, '-payment-token' ) ) {
			$token_val = is_scalar( $v ) ? (string) $v : '';
			if ( $token_val && $token_val !== 'new' ) {
				return;
			}
		}
	}

	// 5) Find the "save payment method" flag.
	$save_val = null;

	// Preferred: exact key for detected gateway id.
	if ( $gateway_id ) {
		$exact = 'wc-' . $gateway_id . '-new-payment-method';
		if ( array_key_exists( $exact, $map ) ) {
			$save_val = $map[ $exact ];
		}
	}

	// Fallback: any "*-new-payment-method" key (covers Stripe + WooPayments variations).
	if ( $save_val === null ) {
		foreach ( $map as $k => $v ) {
			if ( is_string( $k ) && str_ends_with( $k, '-new-payment-method' ) ) {
				$save_val = $v;
				break;
			}
		}
	}

	// Extra Stripe fallbacks (some installs use these IDs)
	if ( $save_val === null ) {
		foreach ( [
			'wc-stripe-new-payment-method',
			'wc-stripe_cc-new-payment-method',
			'wc-wc_stripe-new-payment-method',
		] as $k ) {
			if ( array_key_exists( $k, $map ) ) {
				$save_val = $map[ $k ];
				break;
			}
		}
	}

	$checked = false;
	if ( is_bool( $save_val ) ) {
		$checked = $save_val;
	} elseif ( is_scalar( $save_val ) ) {
		$checked = in_array( strtolower( (string) $save_val ), [ '1', 'true', 'yes', 'on' ], true );
	}

	if ( ! $checked ) {
		if ( class_exists( '\Automattic\WooCommerce\StoreApi\Exceptions\RouteException' ) ) {
			throw new \Automattic\WooCommerce\StoreApi\Exceptions\RouteException(
				'easy_subscriptions_save_payment_required',
				__( 'To purchase a subscription, you must enable “Save payment information to my account for future purchases.”', 'easy-subscriptions' ),
				400
			);
		}

		wc_add_notice(
			__( 'To purchase a subscription, you must enable “Save payment information to my account for future purchases.”', 'easy-subscriptions' ),
			'error'
		);
	}
}, 20, 2 );