<?php
if ( ! defined( 'ABSPATH' ) ) exit;
class Easy_Subscriptions_Gateway_WooPayments implements Easy_Subscriptions_Gateway_Interface {






    // ---- Interface-required stubs ----
    public function charge_initial( $order_id, $amount, $currency, $customer_data ) { return 'wcpay_initial_' . (int) $order_id; }
    public function charge_recurring( $subscription_id, $amount, $currency ) { return $this->charge_recurring_item_unit( $subscription_id, 0, 0, $amount, $currency ); }
    public function cancel_subscription( $subscription_id ) { update_post_meta( $subscription_id, '_easy_subscriptions_status', 'cancelled' ); }





    public function charge_recurring_item_unit( $order_id, $item_id, $unit, $amount, $currency ) {
        $order_id = (int) $order_id;
        $item_id  = (int) $item_id;
        $unit     = (int) $unit;

        $parent_order = wc_get_order( $order_id );
        if ( ! $parent_order ) {
            throw new Exception( "WooPayments: invalid parent order" );
        }

        // ---- lock (prevents concurrent double charge) ----
        $processing_lock_key = '_easy_subscriptions_processing_' . $item_id . '_' . $unit;
        if ( ! add_post_meta( $order_id, $processing_lock_key, time(), true ) ) {
            return '';
        }

        try {
            // ---- Ensure we are using a *WooPayments* token (not Stripe) ----
            $token = $this->get_wcpay_token_strict( $parent_order, $order_id );
            if ( ! $token ) {
                throw new Exception( 'WooPayments: no WooPayments payment token available for this customer.' );
            }

            // ---- Idempotency: per cycle (item+unit+cycle_key) ----
            $cycle_key = wc_get_order_item_meta( $item_id, '_easy_subscriptions_current_cycle_u' . $unit, true );
            if ( empty( $cycle_key ) ) {
                $next_at = wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, true );
                if ( empty( $next_at ) ) $next_at = wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment', true );
                if ( empty( $next_at ) ) $next_at = get_post_meta( $order_id, '_easy_subscriptions_next_payment', true );
                if ( empty( $next_at ) ) $next_at = time();
                $cycle_key = (string) $next_at;
                wc_update_order_item_meta( $item_id, '_easy_subscriptions_current_cycle_u' . $unit, $cycle_key );
            }

            $charged_flag_key = '_easy_subscriptions_cycle_charged_u' . $unit . '_' . $cycle_key . '_i' . $item_id;
            if ( get_post_meta( $order_id, $charged_flag_key, true ) ) {
                return 'skipped';
            }

            // ---- Create renewal order (single unit) ----
            $renewal = wc_create_order( [
                'customer_id' => $parent_order->get_user_id(),
            ] );

            if ( ! $renewal ) {
                throw new Exception( 'WooPayments: could not create renewal order.' );
            }

            $renewal->set_currency( $parent_order->get_currency() );
            $renewal->set_address( $parent_order->get_address( 'billing' ), 'billing' );
            $renewal->set_address( $parent_order->get_address( 'shipping' ), 'shipping' );
            $renewal->set_billing_email( $parent_order->get_billing_email() );
            $renewal->set_billing_phone( $parent_order->get_billing_phone() );

            if ( $item_id ) {
                $orig_item = $parent_order->get_item( $item_id );
                if ( $orig_item ) {
                    $product = $orig_item->get_product();
                    if ( $product ) {
                        $annotated_name = $product->get_name() . sprintf( ' (Recurring from subscription order #%d)', $order_id );
                        $renewal->add_product( $product, 1, [ 'name' => $annotated_name ] );
                        //$renewal->add_product( $product, 1 );
                    }
                }
            }

            $renewal->set_total( (float) $amount );
            $renewal->set_payment_method( 'woocommerce_payments' );
            $renewal->add_payment_token( $token );
            $renewal->save();

            // ---- Actually charge using the real WooPayments gateway ----
            $gateway = $this->get_wcpay_gateway_instance();
            if ( ! $gateway ) {
                throw new Exception( 'WooPayments: gateway instance not available.' );
            }

            // WooPayments expects a "checkout-like" context for tokens
            $this->prime_checkout_context_for_wcpay( $parent_order, $token );

            $result = $gateway->process_payment( $renewal->get_id() );

            // Clean up POST context (best effort)
            $this->unprime_checkout_context_for_wcpay();

            // ---- Validate payment result ----
            if ( ! is_array( $result ) || ( $result['result'] ?? '' ) !== 'success' ) {
                $status = $renewal->get_status();
                throw new Exception( "WooPayments: process_payment did not succeed (renewal status={$status})." );
            }

            // If gateway set a transaction id, keep it; else payment_complete still marks paid if status moved.
            $renewal = wc_get_order( $renewal->get_id() );

            if ( $renewal && in_array( $renewal->get_status(), [ 'processing', 'completed' ], true ) ) {
                // Mark cycle paid
                update_post_meta( $order_id, $charged_flag_key, time() );

                // Note on parent
                $parent_order->add_order_note(
                    sprintf(
                        'Easy Subscriptions (WooPayments): Renewal successful for item %d unit %d. Renewal order #%d. [cycle_key=%s]',
                        $item_id,
                        $unit,
                        $renewal->get_id(),
                        $cycle_key
                    )
                );

                return 'wcpay_success_' . $renewal->get_id();
            }

            throw new Exception( 'WooPayments: renewal order did not move to paid status.' );

        } finally {
            delete_post_meta( $order_id, $processing_lock_key );
        }
    }

    /**
     * STRICT: only accept a token whose gateway_id is exactly 'woocommerce_payments'.
     * Prevents accidentally using WC_Stripe_Payment_Token_CC etc.
     */
    private function get_wcpay_token_strict( WC_Order $order, $order_id ) {
        $user_id = $order->get_user_id();

        // 1) Try token id saved on order meta (but verify it's WCPay)
        $saved = (int) get_post_meta( $order_id, '_wcpay_payment_method_id', true );
        if ( $saved ) {
            $t = WC_Payment_Tokens::get( $saved );
            if ( $t && $t->get_gateway_id() === 'woocommerce_payments' ) {
                return $t;
            }
        }

        // 2) Pull from customer tokens for WooPayments only
        if ( $user_id ) {
            $tokens = WC_Payment_Tokens::get_customer_tokens( $user_id, 'woocommerce_payments' );
            if ( ! empty( $tokens ) ) {
                // prefer default *for this gateway*
                if ( method_exists( 'WC_Payment_Tokens', 'get_customer_default_token' ) ) {
                    $default = WC_Payment_Tokens::get_customer_default_token( $user_id );
                    if ( $default && $default->get_gateway_id() === 'woocommerce_payments' ) {
                        return $default;
                    }
                }

                // otherwise first matching token (should all match since we requested gateway)
                foreach ( $tokens as $t ) {
                    if ( $t && $t->get_gateway_id() === 'woocommerce_payments' ) {
                        // cache it for next time
                        update_post_meta( $order_id, '_wcpay_payment_method_id', (int) $t->get_id() );
                        return $t;
                    }
                }
            }
        }

        return null;
    }

    /** Get the enabled WooPayments gateway instance the same way WooCommerce does. */
    private function get_wcpay_gateway_instance() {
        if ( ! function_exists( 'WC' ) || ! WC()->payment_gateways() ) {
            return null;
        }
        $gateways = WC()->payment_gateways()->payment_gateways();
        return $gateways['woocommerce_payments'] ?? null;
    }

    /**
     * WooPayments process_payment() expects checkout POST fields for token selection.
     * We set them temporarily so Action Scheduler renewals can run.
     */
    private function prime_checkout_context_for_wcpay( WC_Order $parent_order, WC_Payment_Token $token ) {
        // ensure a user context (some gateways check capabilities/session-ish behavior)
        $uid = (int) $parent_order->get_user_id();
        if ( $uid ) {
            wp_set_current_user( $uid );
        }

        // These are the standard token fields for gateways
        $_POST['payment_method'] = 'woocommerce_payments';
        $_POST['wc-woocommerce_payments-payment-token'] = (string) $token->get_id();
        $_POST['wc-woocommerce_payments-new-payment-method'] = 'false';
    }

    /** Undo the temporary POST fields. */
    private function unprime_checkout_context_for_wcpay() {
        unset(
            $_POST['payment_method'],
            $_POST['wc-woocommerce_payments-payment-token'],
            $_POST['wc-woocommerce_payments-new-payment-method']
        );
        wp_set_current_user( 0 );
    }

    
}