<?php
if ( ! defined( 'ABSPATH' ) ) exit;
class Easy_Subscriptions_Gateway_Stripe implements Easy_Subscriptions_Gateway_Interface {
    private $secret_key;
    private $api_base = 'https://api.stripe.com/v1';

    private function init_stripe() {
        if ( ! class_exists( 'WC_Gateway_Stripe' ) ) {
            throw new Exception( 'WooCommerce Stripe Gateway is not active.' );
        }

        // Get the live gateway instance (so settings are loaded correctly)
        $gateways = WC()->payment_gateways()->payment_gateways();
        if ( ! isset( $gateways['stripe'] ) || ! $gateways['stripe'] instanceof WC_Gateway_Stripe ) {
            throw new Exception( 'Stripe gateway not found in WooCommerce.' );
        }

        /** @var WC_Gateway_Stripe $stripe_gateway */
        $stripe_gateway = $gateways['stripe'];

        // Respect test mode
        $testmode   = $stripe_gateway->get_option( 'testmode' );
        $secret_key = ( 'yes' === $testmode )
            ? $stripe_gateway->get_option( 'test_secret_key' )
            : $stripe_gateway->get_option( 'secret_key' );

        if ( empty( $secret_key ) ) {
            throw new Exception( 'Stripe secret key is not configured in WooCommerce.' );
        }

        // Store for REST calls against Stripe; we do NOT rely on stripe-php being present.
        $this->secret_key = $secret_key;
    }

    /**
     * Low-level Stripe API request using WP HTTP API and the key from WooCommerce Stripe settings.
     */
    private function api_request( $method, $endpoint, $params = array(), $idempotency_key = '' ) {
        $url  = rtrim( $this->api_base, '/' ) . '/' . ltrim( $endpoint, '/' );
        $args = array(
            'method'  => strtoupper( $method ),
            'headers' => array(
                'Authorization' => 'Bearer ' . $this->secret_key,
                'Content-Type'  => 'application/x-www-form-urlencoded',
            ),
            'timeout' => 70,
            'body'    => $params,
        );
        if ( ! empty( $idempotency_key ) ) {
            $args['headers']['Idempotency-Key'] = $idempotency_key;
        }

        $response = wp_remote_request( $url, $args );
        if ( is_wp_error( $response ) ) {
            throw new Exception(
                sprintf(
                    'Stripe request error: %s',
                    esc_html( $response->get_error_message() )
                )
            );
        }
        $code = (int) wp_remote_retrieve_response_code( $response );
        $body = json_decode( wp_remote_retrieve_body( $response ), true );

        if ( $code >= 400 ) {
            $message = isset( $body['error']['message'] ) ? $body['error']['message'] : 'Unknown Stripe API error';
               throw new Exception(
                sprintf(
                    'Stripe API error (%d): %s',
                    (int) $code,
                    esc_html( $message )
                )
            );
        }

        return $body;
    }

    private function is_zero_decimal_currency( $currency ) {
        $zero = array( 'bif','clp','djf','gnf','jpy','kmf','krw','mga','pyg','rwf','ugx','vnd','vuv','xaf','xof','xpf' );
        return in_array( strtolower( $currency ), $zero, true );
    }

    private function format_amount_for_stripe( $amount, $currency ) {
        if ( $this->is_zero_decimal_currency( $currency ) ) {
            return (int) round( (float) $amount );
        }
        return (int) round( (float) $amount * 100 );
    }

    public function charge_initial( $order_id, $amount, $currency, $customer_data ) {
        //error_log('EasySubs: charge_initial called for order ' . $order_id . ', PM ID: ' . ($customer_data['payment_method_id'] ?? 'NONE'));
        try {
            $this->init_stripe();

            $order = wc_get_order( $order_id );
            if ( ! $order ) {
                throw new Exception( 'Invalid order.' );
            }

            if ( empty( $customer_data['payment_method_id'] ) ) {
                throw new Exception( 'Missing Stripe payment method id.' );
            }
            $pm_id = $customer_data['payment_method_id'];
            $email = ! empty( $customer_data['email'] ) ? $customer_data['email'] : $order->get_billing_email();

            // Create a Stripe customer if one is not already saved
            $customer_id = get_post_meta( $order_id, '_stripe_customer_id', true );
            if ( empty( $customer_id ) ) {
                $created = $this->api_request( 'POST', 'customers', array( 'email' => $email ) );
                $customer_id = $created['id'];
            }

            // Attach payment method
            $this->api_request( 'POST', 'payment_methods/' . $pm_id . '/attach', array( 'customer' => $customer_id ) );
            $this->api_request( 'POST', 'customers/' . $customer_id, array(
                'invoice_settings[default_payment_method]' => $pm_id,
            ) );

            // First payment
            $amount_int  = $this->format_amount_for_stripe( $amount, $currency );
            $currency_lc = strtolower( $currency );
            $params      = array(
                'amount'                             => $amount_int,
                'currency'                           => $currency_lc,
                'customer'                           => $customer_id,
                'payment_method'                     => $pm_id,
                'off_session'                        => 'false',
                'confirm'                            => 'true',
                'setup_future_usage'                 => 'off_session',
                'automatic_payment_methods[enabled]' => 'true',
                'description'                        => sprintf( 'Initial subscription charge for order #%d', $order_id ),
                'metadata[order_id]'                 => (string) $order_id,
            );

            $intent = $this->api_request( 'POST', 'payment_intents', $params, 'easy_subs_initial_' . $order_id );

            // Save for recurring
            update_post_meta( $order_id, '_stripe_customer_id', $customer_id );
            update_post_meta( $order_id, '_stripe_payment_method_id', $pm_id );
            if ( $order && $order->get_user_id() ) {
                update_user_meta( $order->get_user_id(), '_stripe_customer_id', $customer_id );
            }
            //error_log( 'EasySubs: saved meta on order ' . $order_id . ' and user.' );

            return isset( $intent['id'] ) ? $intent['id'] : '';
        } catch ( \Exception $e ) {
            throw $e;
        }
    }

public function charge_recurring( $subscription_id, $amount, $currency ) {
return 'emtpy';
}



public function charge_recurring_item_unit( $order_id, $item_id, $unit, $amount, $currency ) {
    $order_id = (int) $order_id;
    $item_id  = (int) $item_id;
    $unit     = (int) $unit;

    try {
        $this->init_stripe();

        // -------- processing lock to avoid concurrent double-charging (item+unit) ---------
        $processing_lock_key = '_easy_subscriptions_processing_' . $item_id . '_' . $unit;
        if ( ! add_post_meta( $order_id, $processing_lock_key, time(), true ) ) {
            //error_log( 'EasySubs: charge_recurring_item_unit: lock exists, skipping for #' . $order_id . " item {$item_id} unit {$unit}" );
            return '';
        }

        // --- debug & fallback to WooCommerce Stripe stored data ---------------
        //error_log( 'EasySubs: charge_recurring_item_unit() boot for #' . $order_id . " item {$item_id} unit {$unit}" );
        $customer_id = get_post_meta( $order_id, '_stripe_customer_id', true );
        $pm_id       = get_post_meta( $order_id, '_stripe_payment_method_id', true );
        //error_log( 'EasySubs: loaded from post meta -> customer_id=' . ( $customer_id ?: 'EMPTY' ) . ', pm_id=' . ( $pm_id ?: 'EMPTY' ) );

        if ( empty( $customer_id ) || empty( $pm_id ) ) {
            $order = wc_get_order( $order_id );
            if ( $order ) {
                $user_id = $order->get_user_id();
                //error_log( 'EasySubs: attempting fallback via user tokens. user_id=' . ( $user_id ?: '0' ) );

                if ( empty( $customer_id ) && $user_id ) {
                    foreach ( [ '_stripe_customer_id', '_wc_stripe_customer_id' ] as $k ) {
                        $maybe = get_user_meta( $user_id, $k, true );
                        if ( ! empty( $maybe ) ) { $customer_id = $maybe; break; }
                    }
                }

                if ( empty( $pm_id ) && $user_id && class_exists( 'WC_Payment_Tokens' ) ) {
                    $default_token = method_exists( 'WC_Payment_Tokens', 'get_customer_default_token' )
                        ? \WC_Payment_Tokens::get_customer_default_token( $user_id )
                        : null;
                    if ( $default_token && 'stripe' === $default_token->get_gateway_id() ) {
                        $pm_id = $default_token->get_token();
                    }
                    if ( empty( $pm_id ) ) {
                        $tokens = \WC_Payment_Tokens::get_customer_tokens( $user_id, 'stripe' );
                        if ( ! empty( $tokens ) ) {
                            $first = reset( $tokens );
                            if ( $first ) { $pm_id = $first->get_token(); }
                        }
                    }
                }

                if ( ! empty( $customer_id ) ) { update_post_meta( $order_id, '_stripe_customer_id', $customer_id ); }
                if ( ! empty( $pm_id ) )       { update_post_meta( $order_id, '_stripe_payment_method_id', $pm_id ); }
            }
        }

        if ( empty( $customer_id ) || empty( $pm_id ) ) {
            throw new Exception( 'Missing saved Stripe customer or payment method.' );
        }

        // -------- stable idempotency per billing cycle (item+unit) ------------
        // Cycle key: prefer per-unit next_payment on the item, else item-level, else order-level, else now
        $cycle_key = wc_get_order_item_meta( $item_id, '_easy_subscriptions_current_cycle_u' . $unit, true );
        if ( empty( $cycle_key ) ) {
            $next_at = wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment_u' . $unit, true );
            if ( empty( $next_at ) ) {
                $next_at = wc_get_order_item_meta( $item_id, '_easy_subscriptions_next_payment', true );
            }
            if ( empty( $next_at ) ) {
                $next_at = get_post_meta( $order_id, '_easy_subscriptions_next_payment', true );
            }
            if ( empty( $next_at ) ) { $next_at = time(); }
            $cycle_key = (string) $next_at;
            wc_update_order_item_meta( $item_id, '_easy_subscriptions_current_cycle_u' . $unit, $cycle_key );
        }

        // Per-cycle once-only flag (item+unit)
$charged_flag_key = '_easy_subscriptions_cycle_charged_u' . $unit . '_' . $cycle_key . '_i' . $item_id;

// Only block if we *already* marked this cycle as paid successfully
if ( get_post_meta( $order_id, $charged_flag_key, true ) ) {
    return 'skipped';
}

        $amount_int  = $this->format_amount_for_stripe( $amount, $currency );
        $currency_lc = strtolower( $currency );

        $params = [
            'amount'                             => $amount_int,
            'currency'                           => $currency_lc,
            'customer'                           => $customer_id,
            'payment_method'                     => $pm_id,
            'off_session'                        => 'true',
            'confirm'                            => 'true',
            'automatic_payment_methods[enabled]' => 'true',
            'description'                        => sprintf( 'Recurring subscription charge for subscription #%d', $order_id ),
            'metadata[order_id]'                 => (string) $order_id,
            'metadata[item_id]'                  => (string) $item_id,
            'metadata[unit]'                     => (string) $unit,
            'metadata[cycle_key]'                => (string) $cycle_key,
        ];

        // Idempotency per item+unit+cycle
        $idempotency = 'easy_subs_recurring_' . $order_id . '_i' . $item_id . '_u' . $unit . '_' . $cycle_key;
        $intent = $this->api_request( 'POST', 'payment_intents', $params, $idempotency );

        // ---------- Create a renewal order (only the single item, qty=1) ----------
        $parent_order = wc_get_order( $order_id );
        $new_order    = wc_create_order();

        if ( $parent_order && $new_order ) {
            // Copy addresses & customer
            $new_order->set_address( $parent_order->get_address( 'billing' ), 'billing' );
            $new_order->set_address( $parent_order->get_address( 'shipping' ), 'shipping' );
            $new_order->set_billing_email( $parent_order->get_billing_email() );
            $new_order->set_billing_phone( $parent_order->get_billing_phone() );
            $new_order->set_customer_id( $parent_order->get_user_id() );
            $new_order->set_currency( $parent_order->get_currency() );

            // Add exactly this product (qty=1), annotated
            $orig_item = $parent_order->get_item( $item_id );
            if ( $orig_item ) {
                $product = $orig_item->get_product();
                if ( $product ) {
                    $annotated_name = $product->get_name() . sprintf( ' (Recurring from order #%d)', $order_id);
                    // Force qty=1 for a single-unit renewal line
                    $new_order->add_product( $product, 1, [ 'name' => $annotated_name ] );
                }
            }
        }

        // Set and save totals for this single-unit charge
        $new_order->set_total( $amount );
        $new_order->save();

        // Mark as paid
        $pi_id  = isset( $intent['id'] ) ? $intent['id'] : 'n/a';
        $status = isset( $intent['status'] ) ? $intent['status'] : '';
        $new_order->payment_complete( $pi_id );
        $new_order_id = $new_order->get_id();


        $orig_item = $parent_order ? $parent_order->get_item( $item_id ) : null;
        $pid       = $orig_item ? ( $orig_item->get_variation_id() ?: $orig_item->get_product_id() ) : 0;

        if ( $pid && 'yes' === get_post_meta( $pid, '_easy_subscriptions_separate_shipping', true ) ) {
            $new_order->update_status( 'completed', 'Auto-completed because separate shipping is active.' );
        }

        // ------- Add order note on parent order (as before) -------------------
        $parent = wc_get_order( $order_id );
        if ( $parent ) {
            $pretty_amount = function_exists( 'wc_price' )
                ? wc_price( (float) $amount, [ 'currency' => $currency ] )
                : ( number_format_i18n( (float) $amount, 2 ) . ' ' . $currency );

            if ( 'succeeded' === $status ) {
                $parent->add_order_note(
                    sprintf(
                        'Easy Subscriptions: Renewal successful for item %d unit %d. Charged %s via Stripe (PI: %s). Created renewal order #%d.',
                        $item_id, $unit, $pretty_amount, $pi_id, $new_order_id
                    )
                );
            } elseif ( in_array( $status, [ 'requires_payment_method', 'requires_action', 'requires_confirmation', 'requires_capture', 'canceled' ], true ) ) {
                $parent->add_order_note(
                    sprintf(
                        'Easy Subscriptions: Renewal did not complete for item %d unit %d (Stripe status: %s, PI: %s). Created renewal order #%d.',
                        $item_id, $unit, $status, $pi_id, $new_order_id
                    )
                );
            } else {
                $parent->add_order_note(
                    sprintf(
                        'Easy Subscriptions: Renewal processed for item %d unit %d with status "%s" (PI: %s). Created renewal order #%d.',
                        $item_id, $unit, $status ?: 'unknown', $pi_id, $new_order_id
                    )
                );
            }
        }
        // Mark this cycle as charged successfully
        update_post_meta( $order_id, $charged_flag_key, time() );
        return isset( $intent['id'] ) ? $intent['id'] : '';
    } catch ( \Exception $e ) {
        throw $e;
    } finally {
        // Always release processing lock
        $processing_lock_key = '_easy_subscriptions_processing_' . $item_id . '_' . $unit;
        delete_post_meta( $order_id, $processing_lock_key );
    }
}

    public function cancel_subscription( $subscription_id ) {
        update_post_meta( $subscription_id, '_easy_subscriptions_status', 'cancelled' );
    }

}