<?php
if ( ! defined( 'ABSPATH' ) ) exit;
class Easy_Subscriptions_Gateway_PayPal implements Easy_Subscriptions_Gateway_Interface {
    private $client_id;
    private $secret;
    private $api_base;


    public function __construct() {
        add_action('woocommerce_thankyou', [$this, 'schedule_subscription_check'], 10, 1);
        add_action('easy_subs_check_subscription_status', [$this, 'check_subscription_status'], 10, 2);
        add_filter('cron_schedules', [$this, 'add_ten_minute_cron_interval']);
        add_action( 'woocommerce_before_checkout_form', [ $this, 'easy_subscriptions_display_paypal_disabled_message' ], 10, 1 );
    }



    //schedule a paypal check to see if the sub was canceled on paypal
    public function schedule_subscription_check($order_id) {
        $subscription_id = get_post_meta($order_id, '_paypal_subscription_id', true);

        if (!$subscription_id) {
            //error_log("[EasySubs] No PayPal subscription ID found for order {$order_id}, skipping schedule.");
            return;
        }

        if (!wp_next_scheduled('easy_subs_check_subscription_status', [$order_id, $subscription_id])) {
            wp_schedule_event(time(), 'every_ten_minutes', 'easy_subs_check_subscription_status', [$order_id, $subscription_id]);
            //error_log("[EasySubs] Scheduled subscription check every 10 minutes for order {$order_id}, subscription {$subscription_id}.");
        } else {
            //error_log("[EasySubs] Subscription check already scheduled for order {$order_id}, subscription {$subscription_id}.");
        }
    }



    //display that only paypal only allows one sub at a time
    function easy_subscriptions_display_paypal_disabled_message() {
        if ( is_checkout() && ! is_wc_endpoint_url() ) {
            if ( easy_subscriptions_count_in_cart() > 1 ) {
                echo '<div class="woocommerce-info" style="margin-bottom: 20px;">';
                echo 'PayPal payment methods are unavailable when ordering multiple subscriptions. Please choose an alternative payment method.';
                echo '</div>';
            }
        }
    }


    //add a cron job
    public function add_ten_minute_cron_interval( $schedules ) {
        $schedules['every_ten_minutes'] = [
            'interval' => 600,
            'display'  => __( 'Every 10 Minutes', 'easy-subscriptions' ),
        ];
        return $schedules;
    }


    
    //check if the sub is active
    public function check_subscription_status($order_id, $subscription_id) {
        //error_log("[EasySubs] Running subscription check for order {$order_id}, subscription {$subscription_id}.");

        $token = $this->get_access_token();
        $response = $this->api_request('GET', "v1/billing/subscriptions/{$subscription_id}", [], $token);

        if (empty($response['status'])) {
            //error_log("[EasySubs] No status returned for subscription {$subscription_id}.");
            return;
        }

        $status = strtoupper($response['status']);
        //error_log("[EasySubs] PayPal returned status '{$status}' for subscription {$subscription_id}.");

        if ($status === 'ACTIVE') {
            update_post_meta($order_id, '_paypal_subscription_status', 'active');
            //error_log("[EasySubs] Subscription {$subscription_id} is ACTIVE, metadata updated.");
        } else {
            update_post_meta($order_id, '_paypal_subscription_status', 'cancelled');

            $order = wc_get_order($order_id);
            if ($order) {
                foreach ($order->get_items() as $item_id => $item) {

                    wc_update_order_item_meta($item_id, '_easy_subscriptions_status_unit_' . 1, 'cancelled');
                }
            }

            //error_log("[EasySubs] Subscription {$subscription_id} is CANCELLED. Metadata updated and order items marked cancelled.");

            // Stop future checks
            $timestamp = wp_next_scheduled('easy_subs_check_subscription_status', [$order_id, $subscription_id]);
            if ($timestamp) {
                wp_unschedule_event($timestamp, 'easy_subs_check_subscription_status', [$order_id, $subscription_id]);
                //error_log("[EasySubs] Unscheduling further checks for subscription {$subscription_id}.");
            }
        }
    }



    //initialize paypal
    private function init_paypal() {
        $settings = get_option('woocommerce-ppcp-data-common', []);
        $this->client_id = $settings['client_id'] ?? '';
        $this->secret    = $settings['client_secret'] ?? '';
        $use_sandbox     = !empty($settings['use_sandbox']);
        $this->api_base  = $use_sandbox ? 'https://api-m.sandbox.paypal.com' : 'https://api-m.paypal.com';
    }



    //get paypal api token
    private function get_access_token() {
        $this->init_paypal();

        $response = wp_remote_post("{$this->api_base}/v1/oauth2/token", [
            'headers' => [
                'Authorization' => 'Basic ' . base64_encode("{$this->client_id}:{$this->secret}"),
                'Content-Type'  => 'application/x-www-form-urlencoded',
            ],
            'body'    => 'grant_type=client_credentials',
            'timeout' => 20,
        ]);

        if (is_wp_error($response)) {
            //error_log('[EasySubs][PayPal] OAuth error: ' . $response->get_error_message());
            return '';
        }

        $code = wp_remote_retrieve_response_code($response);
        $raw  = wp_remote_retrieve_body($response);

        if ($code < 200 || $code >= 300) {
            //error_log("[EasySubs][PayPal] OAuth failed ({$code}): {$raw}");
            return '';
        }

        $body = json_decode($raw, true);
        return $body['access_token'] ?? '';
    }



    //run api requests
    private function api_request($method, $endpoint, $params = [], $token = '') {
        $url = "{$this->api_base}/" . ltrim($endpoint, '/');
        $args = [
            'method'  => strtoupper($method),
            'headers' => [
                'Authorization' => 'Bearer ' . $token,
                'Content-Type'  => 'application/json',
            ],
            'body'    => $params ? wp_json_encode($params) : null,
        ];
        $res = wp_remote_request($url, $args);
        return json_decode(wp_remote_retrieve_body($res), true);
    }




    //first paypal charge to create sub
    public function charge_initial($order_id, $amount, $currency, $customer_data) {
        $token = $this->get_access_token();
        $order = wc_get_order($order_id);

        // If already redirected once for this order, skip
        if (get_post_meta($order_id, '_paypal_redirected', true)) {
            return get_post_meta($order_id, '_paypal_subscription_id', true);
        }

        $interval = get_post_meta($order_id, '_easy_subscriptions_period_interval', true) ?: 1;
        $unit     = get_post_meta($order_id, '_easy_subscriptions_period_unit', true) ?: 'MONTH';

        // 1. Create product (only once, reuse later)
        $product_id = $this->ensure_product_id($token);
        if (empty($product_id)) {
            // Something went wrong (token issue, PayPal issue, etc.)
            return;
        }
        


        //free trial logic
        foreach ($order->get_items() as $item) {
            $trial_product_id = $item->get_product_id();
            $variation_id = $item->get_variation_id();
            $post_id = $variation_id ? $variation_id : $trial_product_id;

            $trial_length = (int) get_post_meta($post_id, '_easy_subscriptions_trial_length', true);
            $trial_unit   = get_post_meta($post_id, '_easy_subscriptions_trial_unit', true);
            $ends     = get_post_meta($post_id, '_easy_subscriptions_ends', true);
            $end_date = get_post_meta($post_id, '_easy_subscriptions_end_date', true);
        }
        //free trial logic
        $extra_cycles = $this->easy_subs_convert_trial_to_intervals($trial_length, $trial_unit, $unit);
        $total_cycles = 1 + $extra_cycles;

        //end date logic
        $regular_total_cycles = 0; // infinite by default
        if ($ends === 'date' && !empty($end_date)) {
            $regular_total_cycles = $this->easy_subs_calculate_cycles_until_end(
                current_time('mysql'), // start now
                $end_date,
                $interval,
                $unit
            );
        }


            // 2. Create plan
        $plan = $this->api_request('POST', 'v1/billing/plans', [
            'product_id' => $product_id,
            'name'       => "Plan for Order {$order_id}",
            'billing_cycles' => [
                [
                    // Trial cycle → free for first interval
                    'frequency' => [
                        'interval_unit'  => strtoupper($unit),
                        'interval_count' => (int)$interval,
                    ],
                    'tenure_type' => 'TRIAL',
                    'sequence'    => 1,
                    'total_cycles'=> $total_cycles,
                    'pricing_scheme' => [
                        'fixed_price' => [
                            'value' => '0.00',
                            'currency_code' => strtoupper($currency),
                        ]
                    ]
                ],
                [
                    // Regular billing after trial
                    'frequency' => [
                        'interval_unit'  => strtoupper($unit),
                        'interval_count' => (int)$interval,
                    ],
                    'tenure_type' => 'REGULAR',
                    'sequence'    => 2,
                    'total_cycles'=> $regular_total_cycles, // infinite
                    'pricing_scheme' => [
                        'fixed_price' => [
                            'value' => (string) wc_format_decimal($amount, 2),
                            'currency_code' => strtoupper($currency),
                        ]
                    ]
                ]
            ],
            'payment_preferences' => [
                'auto_bill_outstanding' => true,
                'setup_fee_failure_action' => 'CONTINUE',
                'payment_failure_threshold' => 3
            ]
        ], $token);

        $plan_id = $plan['id'] ?? '';
        if (!$plan_id) {
        // throw new Exception("Failed to create PayPal plan: " . print_r($plan, true));
        return;
        }
        update_post_meta($order_id, '_paypal_plan_id', $plan_id);

        // 3. Create subscription
        $sub = $this->api_request('POST', 'v1/billing/subscriptions', [
            'plan_id' => $plan_id,
            'subscriber' => [
                'email_address' => $customer_data['email'],
            ],
            'application_context' => [
                'brand_name' => get_bloginfo('name'),
                'user_action' => 'SUBSCRIBE_NOW',
                'return_url' => $order->get_checkout_order_received_url(),
                'cancel_url' => $order->get_cancel_order_url(),
            ]
        ], $token);

        if (!empty($sub['id'])) {
            update_post_meta($order_id, '_paypal_subscription_id', $sub['id']);

            // Find approval URL
            if (!empty($sub['links'])) {
                foreach ($sub['links'] as $link) {
                    if ($link['rel'] === 'approve') {
                        // Mark this order so we don’t redirect again
                        update_post_meta($order_id, '_paypal_redirected', 1);

                        // Redirect immediately to PayPal
                        wp_safe_redirect($link['href']);
                        exit;
                    }
                }
            }

        // throw new Exception("Subscription created but no approval URL found: " . print_r($sub, true));
        } else {
            //throw new Exception("Failed to create PayPal subscription: " . print_r($sub, true));
        }
    }



    //green light for other functions
    public function charge_recurring($subscription_id, $amount, $currency) {
        // With PayPal Subscriptions, PayPal does the charging itself.
        // Here we just acknowledge that the subscription is active.
        return 'paypal_auto';
    }


    
    public function charge_recurring_item_unit($order_id, $item_id, $unit, $amount, $currency) {
        return $this->charge_recurring($order_id, $amount, $currency);
    }



    //cancel subscription on paypal with api request
    public function cancel_subscription($subscription_id) {
        $token = $this->get_access_token();
        $this->api_request('POST', "v1/billing/subscriptions/{$subscription_id}/cancel", [
            'reason' => 'Cancelled by user'
        ], $token);
    }



    //free trial value from settings converted to paypal free trial intervals, since subs go in intervals
    function easy_subs_convert_trial_to_intervals($trial_length, $trial_unit, $period_unit) {
        $map = [
            'day'   => 1,
            'week'  => 7,
            'month' => 30,
            'year'  => 365,
        ];

        if (empty($trial_length) || $trial_length < 1) {
            return 0;
        }

        // Convert trial into days
        $trial_days = $trial_length * ($map[$trial_unit] ?? 0);

        // Convert base unit into days
        $unit_days = $map[$period_unit] ?? 30;

        // Round up to full billing cycles
        return (int) ceil($trial_days / $unit_days);
    }



    //calculate cycles for paypal, if an end date is set
    function easy_subs_calculate_cycles_until_end($start_date, $end_date, $interval, $unit) {
        if (empty($end_date)) {
            return 0; // infinite
        }

        $start = new DateTime($start_date);
        $end   = new DateTime($end_date);

        if ($end <= $start) {
            return 0; // already ended or invalid
        }

        $diff_days = $end->diff($start)->days;

        $map = [
            'day'   => 1,
            'week'  => 7,
            'month' => 30,
            'year'  => 365,
        ];

        $unit_days = $map[strtolower($unit)] ?? 30;

        return (int) floor($diff_days / ($interval * $unit_days));
    }


    //refund of subs
    public function refund_capture( $capture_id, $amount, $currency, $note = '' ) {
        $token = $this->get_access_token();

        $params = [
            'amount' => [
                'value'         => wc_format_decimal( $amount, 2 ),
                'currency_code' => strtoupper( $currency ),
            ],
        ];
        if ( $note ) {
            $params['note_to_payer'] = $note;
        }

        return $this->api_request(
            'POST',
            "v2/payments/captures/{$capture_id}/refund",
            $params,
            $token
        );
    }



    //create paypal product id of a paypal sub
    private function ensure_product_id($token) {
        $env = $this->get_paypal_env_key();
        $option_key = "easy_subs_paypal_product_id_{$env}";

        $product_id = get_option($option_key);

        // If we have a cached product, validate it
        if (!empty($product_id)) {
            $check = $this->api_request('GET', "v1/catalogs/products/{$product_id}", [], $token);

            // If PayPal returns RESOURCE_NOT_FOUND, cached ID is invalid for this env/account
            if (
                (isset($check['name']) && $check['name'] === 'RESOURCE_NOT_FOUND') ||
                (isset($check['details'][0]['issue']) && $check['details'][0]['issue'] === 'INVALID_RESOURCE_ID')
            ) {
                //error_log("[EasySubs][PayPal] Cached product_id invalid for {$env}. Recreating. Old: {$product_id}");
                delete_option($option_key);
                $product_id = '';
            }

            // If we got some other error, don't blindly recreate
            if (isset($check['_error'])) {
                //error_log("[EasySubs][PayPal] Product validation failed due to HTTP error; not recreating. {$check['_error']}");
                return '';
            }
        }

        // Create if missing
        if (empty($product_id)) {
            $created = $this->api_request('POST', 'v1/catalogs/products', [
                'name' => 'EasySubs Subscription',
                'type' => 'SERVICE',
            ], $token);

            if (empty($created['id'])) {
                //error_log('[EasySubs][PayPal] Failed to create product. Response: ' . wp_json_encode($created));
                return '';
            }

            $product_id = $created['id'];
            update_option($option_key, $product_id);

            //error_log("[EasySubs][PayPal] Created new product_id for {$env}: {$product_id}");
        }

        return $product_id;
    }


    //is live or is test
    private function get_paypal_env_key() {
        $settings = get_option('woocommerce-ppcp-data-common', []);
        $use_sandbox = !empty($settings['use_sandbox']);
        return $use_sandbox ? 'sandbox' : 'live';
    }
}


new Easy_Subscriptions_Gateway_PayPal();