<?php
/**
 * Plugin Name: Easy Subscriptions
 * Version: 1.0.4
 * Description: Create automatic recurring payments.
 * Author: Appalify
 * Author URI: https://appalify.com/
 * Requires at least: 4.0
 * Tested up to: 6.9
 * Requires Plugins: woocommerce
 * Text Domain: easy-subscriptions
 * Domain Path: /lang/
 * WC requires at least: 8.2
 * WC tested up to: 10.4.3
 *
 * License: GNU General Public License v3.0
 * License URI: http://www.gnu.org/licenses/gpl-3.0.html
 * @package WordPress
 * @author Appalify
 * @since 1.0.0
 */

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

// Load plugin class files.
require_once 'includes/class-easy-subscriptions.php';
require_once 'includes/class-easy-subscriptions-settings.php';

// Load plugin libraries.
require_once 'includes/lib/class-easy-subscriptions-admin-api.php';
require_once 'includes/lib/class-easy-subscriptions-post-type.php';
require_once 'includes/lib/class-easy-subscriptions-taxonomy.php';

add_action( 'plugins_loaded', 'easy_subscriptions_woocommerce_init' );


add_action( 'before_woocommerce_init', function() {
	if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', __FILE__, true );
	}
} );

function easy_subscriptions_woocommerce_init() {
	load_plugin_textdomain( 'easy-subscriptions', false, plugin_basename( dirname( __FILE__ ) ) . '/languages' );

	if ( ! class_exists( 'WooCommerce' ) ) {
		add_action( 'admin_notices', 'easy_subscriptions_missing_wc_notice' );
		return;
	}


	easy_subscriptions();
}

add_action('wp_enqueue_scripts', function () {
    if ( ! is_checkout() ) {
        return;
    }

    // Get Stripe gateway instance safely
    if ( ! class_exists('WC_Payment_Gateways') ) {
        return;
    }
    $gateways = WC()->payment_gateways()->payment_gateways();
    if ( empty($gateways['stripe']) || ! $gateways['stripe'] instanceof WC_Gateway_Stripe ) {
        return;
    }
    /** @var WC_Gateway_Stripe $stripe */
    $stripe   = $gateways['stripe'];
    $testmode = 'yes' === $stripe->get_option('testmode');

    $publishable_key = $testmode
        ? $stripe->get_option('test_publishable_key')
        : $stripe->get_option('publishable_key');

    // Enqueue your script
    wp_enqueue_script(
        'easy-subs-checkout',
        plugins_url('assets/js/easy-subscriptions-checkout.js', __FILE__),
        ['jquery', 'wc-checkout'], // dependencies
        '1.0',
        true
    );


    // Hand the publishable key (and a couple handy flags) to your JS
    wp_localize_script('easy-subs-checkout', 'easy_subs_params', [
        'publishableKey' => $publishable_key,
        'testmode'       => $testmode ? 'yes' : 'no',
    ]);
});


//plugin url path
if (!defined('EASY_SUBS_PLUGIN_URL')) {
    define('EASY_SUBS_PLUGIN_URL', plugin_dir_url(__FILE__)); // URL to the root of the plugin
}

if (!defined('EASY_SUBS_PLUGIN_DIR')) {
    define('EASY_SUBS_PLUGIN_DIR', plugin_dir_path(__FILE__)); // Absolute path to the plugin directory
}

function easy_subscriptions_missing_wc_notice() {
	$install_url = wp_nonce_url(
		add_query_arg(
			[
				'action' => 'install-plugin',
				'plugin' => 'woocommerce',
			],
			admin_url( 'update.php' )
		),
		'install-plugin_woocommerce'
	);

	$admin_notice_content = sprintf(
		// translators: 1$-2$: opening and closing <strong> tags, 3$-4$: link tags, takes to woocommerce plugin on wp.org, 5$-6$: opening and closing link tags, leads to plugins.php in admin
		esc_html__( '%2$s The %3$sWooCommerce plugin%4$s must be active for your plugin to work. Please %5$sinstall & activate WooCommerce &raquo;%6$s', 'easy-subscriptions' ),
		'<strong>',
		'</strong>',
		'<a href="http://wordpress.org/extend/plugins/woocommerce/">',
		'</a>',
		'<a href="' . esc_url( $install_url ) . '">',
		'</a>'
	);

	echo '<div class="error">';
	echo '<p>' . wp_kses_post( $admin_notice_content ) . '</p>';
	echo '</div>';
}

	


/**
 * Returns the main instance of easy_subscriptions to prevent the need to use globals.
 *
 * @since  1.0.0
 * @return object easy_subscriptions
 */
function easy_subscriptions() {
	$instance = easy_subscriptions::instance( __FILE__, '1.0.4' );

	if ( is_null( $instance->settings ) ) {
		$instance->settings = easy_subscriptions_Settings::instance( $instance );
	}

	return $instance;
}
